/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.wicket.settings;

import org.apache.wicket.markup.MarkupFactory;

/**
 * Interface for markup related settings.
 * <p>
 * <i>compressWhitespace </i> (defaults to false) - Causes pages to render with redundant whitespace
 * removed. Whitespace stripping is not HTML or JavaScript savvy and can conceivably break pages,
 * but should provide significant performance improvements.
 * <p>
 * <i>stripComments</i> (defaults to false) - Set to true to strip HTML comments during markup
 * loading
 * 
 * @author Igor Vaynberg (ivaynberg)
 */
public interface IMarkupSettings
{
	/**
	 * If true, automatic link resolution is enabled. Disabled by default.
	 * 
	 * @see org.apache.wicket.markup.resolver.AutoLinkResolver
	 * @see org.apache.wicket.markup.parser.filter.WicketLinkTagHandler
	 * @return Returns the automaticLinking.
	 */
	boolean getAutomaticLinking();

	/**
	 * @return Returns the compressWhitespace.
	 * @see IMarkupSettings#setCompressWhitespace(boolean)
	 */
	boolean getCompressWhitespace();

	/**
	 * @return Returns the defaultAfterDisabledLink.
	 */
	String getDefaultAfterDisabledLink();

	/**
	 * @return Returns the defaultBeforeDisabledLink.
	 */
	String getDefaultBeforeDisabledLink();

	/**
	 * @since 1.1
	 * @return Returns default encoding of markup files. If null, the operating system provided
	 *         encoding will be used.
	 */
	String getDefaultMarkupEncoding();

	/**
	 * @return Returns the stripComments.
	 * @see IMarkupSettings#setStripComments(boolean)
	 */
	boolean getStripComments();

	/**
	 * Gets whether to remove wicket tags from the output.
	 * 
	 * @return whether to remove wicket tags from the output
	 */
	boolean getStripWicketTags();

	/**
	 * @since 1.3
	 * @return if true, an exception is thrown if the markup file does not contain a xml declaration
	 */
	boolean getThrowExceptionOnMissingXmlDeclaration();

	/**
	 * Application default for automatic link resolution.
	 * 
	 * @param automaticLinking
	 *            The automaticLinking to set.
	 * @see org.apache.wicket.markup.resolver.AutoLinkResolver
	 * @see org.apache.wicket.markup.parser.filter.WicketLinkTagHandler
	 */
	void setAutomaticLinking(boolean automaticLinking);

	/**
	 * Turns on whitespace compression. Multiple occurrences of space/tab characters will be
	 * compressed to a single space. Multiple line breaks newline/carriage-return will also be
	 * compressed to a single newline.
	 * <p>
	 * Compression is currently not HTML aware and so it may be possible for whitespace compression
	 * to break pages. For this reason, whitespace compression is off by default and you should test
	 * your application thoroughly after turning whitespace compression on.
	 * <p>
	 * Spaces are removed from markup at markup load time and there should be no effect on page
	 * rendering speed. In fact, your pages should render faster with whitespace compression
	 * enabled.
	 * 
	 * @param compressWhitespace
	 *            The compressWhitespace to set.
	 */
	void setCompressWhitespace(final boolean compressWhitespace);

	/**
	 * @param defaultAfterDisabledLink
	 *            The defaultAfterDisabledLink to set.
	 */
	void setDefaultAfterDisabledLink(String defaultAfterDisabledLink);

	/**
	 * @param defaultBeforeDisabledLink
	 *            The defaultBeforeDisabledLink to set.
	 */
	void setDefaultBeforeDisabledLink(String defaultBeforeDisabledLink);

	/**
	 * Set default encoding for markup files. If null, the encoding provided by the operating system
	 * will be used.
	 * 
	 * @since 1.1
	 * @param encoding
	 */
	void setDefaultMarkupEncoding(final String encoding);

	/**
	 * Enables stripping of markup comments denoted in markup by HTML comment tagging.
	 * 
	 * @param stripComments
	 *            True to strip markup comments from rendered pages
	 */
	void setStripComments(boolean stripComments);

	/**
	 * Sets whether to remove wicket tags from the output.
	 * 
	 * @param stripWicketTags
	 *            whether to remove wicket tags from the output
	 */
	void setStripWicketTags(boolean stripWicketTags);

	/**
	 * If true, an exception is thrown if the markup file does not contain a xml declaration
	 * 
	 * @since 1.3
	 * @param throwException
	 */
	void setThrowExceptionOnMissingXmlDeclaration(final boolean throwException);

	/**
	 * Get the markup factory
	 * 
	 * @return A new instance of MarkupFactory.
	 */
	MarkupFactory getMarkupFactory();

	/**
	 * Set a new markup factory
	 * 
	 * @param factory
	 */
	void setMarkupFactory(MarkupFactory factory);
}
