/*
 * Decompiled with CFR 0.152.
 */
package org.apache.uniffle.server;

import io.prometheus.client.CollectorRegistry;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.collections.CollectionUtils;
import org.apache.uniffle.common.Arguments;
import org.apache.uniffle.common.ClientType;
import org.apache.uniffle.common.ServerStatus;
import org.apache.uniffle.common.StorageType;
import org.apache.uniffle.common.config.RssBaseConf;
import org.apache.uniffle.common.config.RssConf;
import org.apache.uniffle.common.exception.InvalidRequestException;
import org.apache.uniffle.common.exception.RssException;
import org.apache.uniffle.common.metrics.GRPCMetrics;
import org.apache.uniffle.common.metrics.JvmMetrics;
import org.apache.uniffle.common.metrics.MetricReporter;
import org.apache.uniffle.common.metrics.MetricReporterFactory;
import org.apache.uniffle.common.metrics.NettyMetrics;
import org.apache.uniffle.common.rpc.ServerInterface;
import org.apache.uniffle.common.security.SecurityConfig;
import org.apache.uniffle.common.security.SecurityContextFactory;
import org.apache.uniffle.common.util.ExitUtils;
import org.apache.uniffle.common.util.RssUtils;
import org.apache.uniffle.common.util.ThreadUtils;
import org.apache.uniffle.common.web.CoalescedCollectorRegistry;
import org.apache.uniffle.common.web.JettyServer;
import org.apache.uniffle.guava.annotations.VisibleForTesting;
import org.apache.uniffle.guava.collect.Lists;
import org.apache.uniffle.guava.collect.Sets;
import org.apache.uniffle.server.Checker;
import org.apache.uniffle.server.HealthCheck;
import org.apache.uniffle.server.RegisterHeartBeat;
import org.apache.uniffle.server.ShuffleFlushManager;
import org.apache.uniffle.server.ShuffleServerConf;
import org.apache.uniffle.server.ShuffleServerFactory;
import org.apache.uniffle.server.ShuffleServerGrpcMetrics;
import org.apache.uniffle.server.ShuffleServerMetrics;
import org.apache.uniffle.server.ShuffleServerNettyMetrics;
import org.apache.uniffle.server.ShuffleTaskManager;
import org.apache.uniffle.server.buffer.ShuffleBufferManager;
import org.apache.uniffle.server.netty.StreamServer;
import org.apache.uniffle.server.storage.StorageManager;
import org.apache.uniffle.server.storage.StorageManagerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import picocli.CommandLine;

public class ShuffleServer {
    private static final Logger LOG = LoggerFactory.getLogger(ShuffleServer.class);
    private RegisterHeartBeat registerHeartBeat;
    private String id;
    private String ip;
    private int grpcPort;
    private int nettyPort;
    private ShuffleServerConf shuffleServerConf;
    private JettyServer jettyServer;
    private ShuffleTaskManager shuffleTaskManager;
    private ServerInterface server;
    private ShuffleFlushManager shuffleFlushManager;
    private ShuffleBufferManager shuffleBufferManager;
    private StorageManager storageManager;
    private HealthCheck healthCheck;
    private Set<String> tags = Sets.newHashSet();
    private GRPCMetrics grpcMetrics;
    private NettyMetrics nettyMetrics;
    private MetricReporter metricReporter;
    private AtomicReference<ServerStatus> serverStatus = new AtomicReference<ServerStatus>(ServerStatus.ACTIVE);
    private volatile boolean running;
    private ExecutorService executorService;
    private Future<?> decommissionFuture;
    private boolean nettyServerEnabled;
    private StreamServer streamServer;

    public ShuffleServer(ShuffleServerConf shuffleServerConf) throws Exception {
        this.shuffleServerConf = shuffleServerConf;
        try {
            this.initialization();
        }
        catch (Exception e) {
            LOG.error("Errors on initializing shuffle server.", (Throwable)e);
            throw e;
        }
    }

    public static void main(String[] args) throws Exception {
        Arguments arguments = new Arguments();
        CommandLine commandLine = new CommandLine((Object)arguments);
        commandLine.parseArgs(args);
        String configFile = arguments.getConfigFile();
        LOG.info("Start to init shuffle server using config {}", (Object)configFile);
        ShuffleServerConf shuffleServerConf = new ShuffleServerConf(configFile);
        ShuffleServer shuffleServer = new ShuffleServer(shuffleServerConf);
        shuffleServer.start();
        shuffleServer.blockUntilShutdown();
    }

    public void start() throws Exception {
        this.jettyServer.start();
        this.grpcPort = this.server.start();
        if (this.nettyServerEnabled) {
            this.nettyPort = this.streamServer.start();
        }
        this.id = this.nettyServerEnabled ? this.ip + "-" + this.grpcPort + "-" + this.nettyPort : this.ip + "-" + this.grpcPort;
        this.shuffleServerConf.setString("rss.server.id", this.id);
        LOG.info("Start to shuffle server with id {}", (Object)this.id);
        this.initMetricsReporter();
        this.registerHeartBeat.startHeartBeat();
        Runtime.getRuntime().addShutdownHook(new Thread(){

            @Override
            public void run() {
                LOG.info("*** shutting down gRPC server since JVM is shutting down");
                try {
                    ShuffleServer.this.stopServer();
                }
                catch (Exception e) {
                    LOG.error(e.getMessage());
                }
                LOG.info("*** server shut down");
            }
        });
        this.running = true;
        LOG.info("Shuffle server start successfully!");
    }

    public void stopServer() throws Exception {
        if (this.jettyServer != null) {
            this.jettyServer.stop();
            LOG.info("Jetty Server Stopped!");
        }
        if (this.registerHeartBeat != null) {
            this.registerHeartBeat.shutdown();
            LOG.info("HeartBeat Stopped!");
        }
        if (this.storageManager != null) {
            this.storageManager.stop();
            LOG.info("MultiStorage Stopped!");
        }
        if (this.healthCheck != null) {
            this.healthCheck.stop();
            LOG.info("HealthCheck stopped!");
        }
        if (this.metricReporter != null) {
            this.metricReporter.stop();
            LOG.info("Metric Reporter Stopped!");
        }
        SecurityContextFactory.get().getSecurityContext().close();
        this.server.stop();
        if (this.nettyServerEnabled && this.streamServer != null) {
            this.streamServer.stop();
        }
        if (this.executorService != null) {
            this.executorService.shutdownNow();
        }
        this.running = false;
        LOG.info("RPC Server Stopped!");
    }

    private void initialization() throws Exception {
        boolean testMode = this.shuffleServerConf.getBoolean(RssBaseConf.RSS_TEST_MODE_ENABLE);
        String storageType = ((StorageType)this.shuffleServerConf.get(RssBaseConf.RSS_STORAGE_TYPE)).name();
        if (!testMode && (org.apache.uniffle.storage.util.StorageType.LOCALFILE.name().equals(storageType) || org.apache.uniffle.storage.util.StorageType.HDFS.name().equals(storageType))) {
            throw new IllegalArgumentException("RSS storage type about LOCALFILE and HADOOP should be used in test mode, because of the poor performance of these two types.");
        }
        this.ip = RssUtils.getHostIp();
        if (this.ip == null) {
            throw new RssException("Couldn't acquire host Ip");
        }
        this.grpcPort = this.shuffleServerConf.getInteger(ShuffleServerConf.RPC_SERVER_PORT);
        this.nettyPort = this.shuffleServerConf.getInteger(ShuffleServerConf.NETTY_SERVER_PORT);
        this.jettyServer = new JettyServer((RssBaseConf)this.shuffleServerConf);
        this.registerMetrics();
        this.jettyServer.addResourcePackages(new String[]{"org.apache.uniffle.common.web.resource"});
        this.jettyServer.registerInstance(CollectorRegistry.class.getCanonicalName() + "#server", (Object)ShuffleServerMetrics.getCollectorRegistry());
        this.jettyServer.registerInstance(CollectorRegistry.class.getCanonicalName() + "#grpc", (Object)this.grpcMetrics.getCollectorRegistry());
        this.jettyServer.registerInstance(CollectorRegistry.class.getCanonicalName() + "#netty", (Object)this.nettyMetrics.getCollectorRegistry());
        this.jettyServer.registerInstance(CollectorRegistry.class.getCanonicalName() + "#jvm", (Object)JvmMetrics.getCollectorRegistry());
        this.jettyServer.registerInstance(CollectorRegistry.class.getCanonicalName() + "#all", (Object)new CoalescedCollectorRegistry(new CollectorRegistry[]{ShuffleServerMetrics.getCollectorRegistry(), this.grpcMetrics.getCollectorRegistry(), this.nettyMetrics.getCollectorRegistry(), JvmMetrics.getCollectorRegistry()}));
        SecurityConfig securityConfig = null;
        if (this.shuffleServerConf.getBoolean(RssBaseConf.RSS_SECURITY_HADOOP_KERBEROS_ENABLE)) {
            securityConfig = SecurityConfig.newBuilder().krb5ConfPath(this.shuffleServerConf.getString(RssBaseConf.RSS_SECURITY_HADOOP_KRB5_CONF_FILE)).keytabFilePath(this.shuffleServerConf.getString(RssBaseConf.RSS_SECURITY_HADOOP_KERBEROS_KEYTAB_FILE)).principal(this.shuffleServerConf.getString(RssBaseConf.RSS_SECURITY_HADOOP_KERBEROS_PRINCIPAL)).reloginIntervalSec(this.shuffleServerConf.getLong(RssBaseConf.RSS_SECURITY_HADOOP_KERBEROS_RELOGIN_INTERVAL_SEC)).build();
        }
        SecurityContextFactory.get().init(securityConfig);
        this.storageManager = StorageManagerFactory.getInstance().createStorageManager(this.shuffleServerConf);
        this.storageManager.start();
        boolean healthCheckEnable = this.shuffleServerConf.getBoolean(ShuffleServerConf.HEALTH_CHECK_ENABLE);
        if (healthCheckEnable) {
            ArrayList<Checker> builtInCheckers = Lists.newArrayList();
            builtInCheckers.add(this.storageManager.getStorageChecker());
            this.healthCheck = new HealthCheck(this.serverStatus, this.shuffleServerConf, builtInCheckers);
            this.healthCheck.start();
        }
        this.registerHeartBeat = new RegisterHeartBeat(this);
        this.shuffleFlushManager = new ShuffleFlushManager(this.shuffleServerConf, this, this.storageManager);
        this.shuffleBufferManager = new ShuffleBufferManager(this.shuffleServerConf, this.shuffleFlushManager);
        this.shuffleTaskManager = new ShuffleTaskManager(this.shuffleServerConf, this.shuffleFlushManager, this.shuffleBufferManager, this.storageManager);
        boolean bl = this.nettyServerEnabled = (Integer)this.shuffleServerConf.get(ShuffleServerConf.NETTY_SERVER_PORT) >= 0;
        if (this.nettyServerEnabled) {
            this.streamServer = new StreamServer(this);
        }
        this.setServer();
        this.initServerTags();
    }

    private void initServerTags() {
        this.tags.add("ss_v4");
        List configuredTags = (List)this.shuffleServerConf.get(ShuffleServerConf.TAGS);
        if (CollectionUtils.isNotEmpty((Collection)configuredTags)) {
            this.tags.addAll(configuredTags);
        }
        this.tagServer();
        LOG.info("Server tags: {}", this.tags);
    }

    private void tagServer() {
        if (this.nettyServerEnabled) {
            this.tags.add(ClientType.GRPC_NETTY.name());
        } else {
            this.tags.add(ClientType.GRPC.name());
        }
    }

    private void registerMetrics() {
        LOG.info("Register metrics");
        CollectorRegistry shuffleServerCollectorRegistry = new CollectorRegistry(true);
        String tags = this.coverToString();
        ShuffleServerMetrics.register(shuffleServerCollectorRegistry, tags);
        this.grpcMetrics = new ShuffleServerGrpcMetrics(tags);
        this.grpcMetrics.register(new CollectorRegistry(true));
        this.nettyMetrics = new ShuffleServerNettyMetrics(tags);
        this.nettyMetrics.register(new CollectorRegistry(true));
        CollectorRegistry jvmCollectorRegistry = new CollectorRegistry(true);
        boolean verbose = this.shuffleServerConf.getBoolean(ShuffleServerConf.RSS_JVM_METRICS_VERBOSE_ENABLE);
        JvmMetrics.register((CollectorRegistry)jvmCollectorRegistry, (boolean)verbose);
    }

    private void initMetricsReporter() throws Exception {
        this.metricReporter = MetricReporterFactory.getMetricReporter((RssConf)this.shuffleServerConf, (String)this.id);
        if (this.metricReporter != null) {
            this.metricReporter.addCollectorRegistry(ShuffleServerMetrics.getCollectorRegistry());
            this.metricReporter.addCollectorRegistry(this.grpcMetrics.getCollectorRegistry());
            this.metricReporter.addCollectorRegistry(JvmMetrics.getCollectorRegistry());
            this.metricReporter.start();
        }
    }

    private void blockUntilShutdown() throws InterruptedException {
        this.server.blockUntilShutdown();
    }

    public ServerStatus getServerStatus() {
        return this.serverStatus.get();
    }

    public void setServerStatus(ServerStatus serverStatus) {
        this.serverStatus.set(serverStatus);
    }

    public synchronized void decommission() {
        if (this.isDecommissioning()) {
            LOG.info("Shuffle Server is decommissioning. Nothing needs to be done.");
            return;
        }
        if (!ServerStatus.ACTIVE.equals((Object)this.serverStatus.get())) {
            throw new InvalidRequestException("Shuffle Server is processing other procedures, current status:" + this.serverStatus);
        }
        this.serverStatus.set(ServerStatus.DECOMMISSIONING);
        LOG.info("Shuffle Server is decommissioning.");
        if (this.executorService == null) {
            this.executorService = ThreadUtils.getDaemonSingleThreadExecutor((String)"shuffle-server-decommission");
        }
        this.decommissionFuture = this.executorService.submit(this::waitDecommissionFinish);
    }

    private void waitDecommissionFinish() {
        int remainApplicationNum;
        long checkInterval = (Long)this.shuffleServerConf.get(ShuffleServerConf.SERVER_DECOMMISSION_CHECK_INTERVAL);
        boolean shutdownAfterDecommission = (Boolean)this.shuffleServerConf.get(ShuffleServerConf.SERVER_DECOMMISSION_SHUTDOWN);
        while (this.isDecommissioning()) {
            remainApplicationNum = this.shuffleTaskManager.getAppIds().size();
            if (remainApplicationNum == 0) {
                this.serverStatus.set(ServerStatus.DECOMMISSIONED);
                LOG.info("All applications finished. Current status is " + this.serverStatus);
                if (!shutdownAfterDecommission) break;
                LOG.info("Exiting...");
                try {
                    this.stopServer();
                }
                catch (Exception e) {
                    ExitUtils.terminate((int)1, (String)"Stop server failed!", (Throwable)e, (Logger)LOG);
                }
                break;
            }
            LOG.info("Shuffle server is decommissioning, remaining {} applications not finished.", (Object)remainApplicationNum);
            try {
                Thread.sleep(checkInterval);
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while waiting for decommission to finish");
                break;
            }
        }
        if ((remainApplicationNum = this.shuffleTaskManager.getAppIds().size()) > 0) {
            LOG.info("Decommission exiting, remaining {} applications not finished.", (Object)remainApplicationNum);
        }
    }

    public synchronized void cancelDecommission() {
        if (!this.isDecommissioning()) {
            LOG.info("Shuffle server is not decommissioning. Nothing needs to be done.");
            return;
        }
        if (ServerStatus.DECOMMISSIONED.equals((Object)this.serverStatus.get())) {
            this.serverStatus.set(ServerStatus.ACTIVE);
            return;
        }
        this.serverStatus.set(ServerStatus.ACTIVE);
        if (this.decommissionFuture.cancel(true)) {
            LOG.info("Decommission canceled.");
        } else {
            LOG.warn("Failed to cancel decommission.");
        }
        this.decommissionFuture = null;
    }

    public String getIp() {
        return this.ip;
    }

    public String getId() {
        return this.id;
    }

    public int getGrpcPort() {
        return this.grpcPort;
    }

    public ShuffleServerConf getShuffleServerConf() {
        return this.shuffleServerConf;
    }

    public ServerInterface getServer() {
        return this.server;
    }

    @VisibleForTesting
    public void setServer() {
        ShuffleServerFactory shuffleServerFactory = new ShuffleServerFactory(this);
        this.server = shuffleServerFactory.getServer();
    }

    public void setServer(ServerInterface server) {
        this.server = server;
    }

    public ShuffleTaskManager getShuffleTaskManager() {
        return this.shuffleTaskManager;
    }

    public ShuffleFlushManager getShuffleFlushManager() {
        return this.shuffleFlushManager;
    }

    public long getUsedMemory() {
        return this.shuffleBufferManager.getUsedMemory();
    }

    public long getPreAllocatedMemory() {
        return this.shuffleBufferManager.getPreAllocatedSize();
    }

    public long getAvailableMemory() {
        return this.shuffleBufferManager.getCapacity() - this.shuffleBufferManager.getUsedMemory();
    }

    public int getEventNumInFlush() {
        return this.shuffleFlushManager.getEventNumInFlush();
    }

    public ShuffleBufferManager getShuffleBufferManager() {
        return this.shuffleBufferManager;
    }

    public StorageManager getStorageManager() {
        return this.storageManager;
    }

    public Set<String> getTags() {
        return Collections.unmodifiableSet(this.tags);
    }

    @VisibleForTesting
    public void markUnhealthy() {
        this.serverStatus.set(ServerStatus.UNHEALTHY);
    }

    public GRPCMetrics getGrpcMetrics() {
        return this.grpcMetrics;
    }

    public NettyMetrics getNettyMetrics() {
        return this.nettyMetrics;
    }

    public boolean isDecommissioning() {
        return ServerStatus.DECOMMISSIONING.equals((Object)this.serverStatus.get()) || ServerStatus.DECOMMISSIONED.equals((Object)this.serverStatus.get());
    }

    @VisibleForTesting
    public boolean isRunning() {
        return this.running;
    }

    public int getNettyPort() {
        return this.nettyPort;
    }

    public String coverToString() {
        List tags = (List)this.shuffleServerConf.get(ShuffleServerConf.TAGS);
        StringBuilder sb = new StringBuilder();
        sb.append("ss_v4");
        if (tags == null || tags.size() == 0) {
            return sb.toString();
        }
        for (String tag : tags) {
            sb.append(",");
            sb.append(tag);
        }
        return sb.toString();
    }
}

