package org.apache.stratum.messenger;

/*
 * Copyright 2001-2005 The Apache Software Foundation or its licensors,
 * as applicable.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.commons.configuration.Configuration;
import org.apache.commons.messenger.MessengerManager;
import org.apache.stratum.lifecycle.Configurable;
import org.apache.stratum.lifecycle.Initializable;

/**
 * This class is the Messenger component.  It is an adaptor that allows the ComponentLoader to startup MessengerManager.  Client
 * code wishing to use Messenger should ask MessengerManager directly for Messenger instances.
 *
 * @author <a href="mailto:eric@dobbse.net">Eric Dobbs</a>
 * @version $Id: MessengerComponent.java 264191 2005-08-29 18:07:52Z henning $
 *
 * @see org.apache.commons.messenger.MessengerManager
 * @see org.apache.commons.messenger.Messenger
 */
public class MessengerComponent
        implements Configurable, Initializable
{
    /** The key for property 'messenger.xml.url' */
    private static final String MESSENGER_XML_URL = "messenger.xml.url";

    /** The location of the Messenger.xml file */
    private String messengerXmlUrl;

    /**
     * Configure MessengerManager.  MessengerManager requires a URL to a Messenger.xml file.  The messenger.properties file should
     * define one property named 'messenger.xml.url' identifying the location of Messenger.xml.
     *
     * @param configuration Configuration object containing the 'messenger.xml.url' property
     */
    public void configure(Configuration configuration)
    {
        messengerXmlUrl = configuration.getString(MESSENGER_XML_URL);
    }

    /**
     * Initialize the MessengerManager.
     *
     * @exception Exception is thrown if the file defined in 'messenger.xml.url' can't be found.  JMSException will be thrown if
     *            MessengerManager.configure is unhappy with the value from 'messenger.xml.url'
     */
    public void initialize()
            throws Exception
    {
        if (messengerXmlUrl == null)
        {
            throw new Exception("Can't find property '" + MESSENGER_XML_URL + "' in the properties file");
        }

        MessengerManager.configure(messengerXmlUrl);
    }
}
