/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.elasticjob.infra.handler.sharding.impl;

import org.apache.shardingsphere.elasticjob.infra.handler.sharding.JobInstance;
import org.apache.shardingsphere.elasticjob.infra.handler.sharding.JobShardingStrategy;
import org.junit.jupiter.api.Test;

import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;

class AverageAllocationJobShardingStrategyTest {
    
    private final JobShardingStrategy jobShardingStrategy = new AverageAllocationJobShardingStrategy();
    
    @Test
    void shardingForZeroServer() {
        assertThat(jobShardingStrategy.sharding(Collections.emptyList(), "test_job", 3), is(Collections.<JobInstance, List<Integer>>emptyMap()));
    }
    
    @Test
    void shardingForOneServer() {
        Map<JobInstance, List<Integer>> expected = new LinkedHashMap<>(1, 1);
        expected.put(new JobInstance("host0@-@0"), Arrays.asList(0, 1, 2));
        assertThat(jobShardingStrategy.sharding(Collections.singletonList(new JobInstance("host0@-@0")), "test_job", 3), is(expected));
    }
    
    @Test
    void shardingForServersMoreThanShardingCount() {
        Map<JobInstance, List<Integer>> expected = new LinkedHashMap<>(3, 1);
        expected.put(new JobInstance("host0@-@0"), Collections.singletonList(0));
        expected.put(new JobInstance("host1@-@0"), Collections.singletonList(1));
        expected.put(new JobInstance("host2@-@0"), Collections.emptyList());
        assertThat(jobShardingStrategy.sharding(Arrays.asList(new JobInstance("host0@-@0"), new JobInstance("host1@-@0"), new JobInstance("host2@-@0")), "test_job", 2), is(expected));
    }
    
    @Test
    void shardingForServersLessThanShardingCountAliquot() {
        Map<JobInstance, List<Integer>> expected = new LinkedHashMap<>(3, 1);
        expected.put(new JobInstance("host0@-@0"), Arrays.asList(0, 1, 2));
        expected.put(new JobInstance("host1@-@0"), Arrays.asList(3, 4, 5));
        expected.put(new JobInstance("host2@-@0"), Arrays.asList(6, 7, 8));
        assertThat(jobShardingStrategy.sharding(Arrays.asList(new JobInstance("host0@-@0"), new JobInstance("host1@-@0"), new JobInstance("host2@-@0")), "test_job", 9), is(expected));
    }
    
    @Test
    void shardingForServersLessThanShardingCountAliquantFor8ShardingCountAnd3Servers() {
        Map<JobInstance, List<Integer>> expected = new LinkedHashMap<>(3, 1);
        expected.put(new JobInstance("host0@-@0"), Arrays.asList(0, 1, 6));
        expected.put(new JobInstance("host1@-@0"), Arrays.asList(2, 3, 7));
        expected.put(new JobInstance("host2@-@0"), Arrays.asList(4, 5));
        assertThat(jobShardingStrategy.sharding(Arrays.asList(new JobInstance("host0@-@0"), new JobInstance("host1@-@0"), new JobInstance("host2@-@0")), "test_job", 8), is(expected));
    }
    
    @Test
    void shardingForServersLessThanShardingCountAliquantFor10ShardingCountAnd3Servers() {
        Map<JobInstance, List<Integer>> expected = new LinkedHashMap<>(3, 1);
        expected.put(new JobInstance("host0@-@0"), Arrays.asList(0, 1, 2, 9));
        expected.put(new JobInstance("host1@-@0"), Arrays.asList(3, 4, 5));
        expected.put(new JobInstance("host2@-@0"), Arrays.asList(6, 7, 8));
        assertThat(jobShardingStrategy.sharding(Arrays.asList(new JobInstance("host0@-@0"), new JobInstance("host1@-@0"), new JobInstance("host2@-@0")), "test_job", 10), is(expected));
    }
}
