/****************************************************************************
 * net/procfs/procfs.h
 *
 *   Copyright (C) 2015 Gregory Nutt. All rights reserved.
 *   Author: Gregory Nutt <gnutt@nuttx.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name NuttX nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

#ifndef __NET_PROCFS_PROCFS_H
#define __NET_PROCFS_PROCFS_H

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>
#include <sys/types.h>
#include <nuttx/fs/procfs.h>

#if defined(CONFIG_FS_PROCFS) && !defined(CONFIG_FS_PROCFS_EXCLUDE_NET)

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

#ifdef CONFIG_FS_PROCFS_EXCLUDE_ROUTE
#  undef CONFIG_NET_ROUTE
#endif

/* Determines the size of an intermediate buffer that must be large enough
 * to handle the longest line generated by this logic.
 */

#define NET_LINELEN 80

/****************************************************************************
 * Public Type Definitions
 ****************************************************************************/

/* Describes the /net directory entries */

enum netprocfs_entry_e
{
  NETPROCFS_SUBDIR_DEV = 0           /* Multiple instances, e.g. /proc/net/eth0 */
#ifdef CONFIG_NET_STATISTICS
  , NETPROCFS_SUBDIR_STAT            /* /proc/net/stat */
#ifdef CONFIG_NET_MLD
  , NETPROCFS_SUBDIR_MLD             /* /proc/net/mld */
#endif
#endif
#ifdef CONFIG_NET_ROUTE
  , NETPROCFS_SUBDIR_ROUTE           /* /proc/net/route */
#endif
};

/* This structure describes one open "file" */

struct net_driver_s;                 /* Forward reference */
struct netprocfs_file_s
{
  struct procfs_file_s base;         /* Base open file structure */
  FAR struct net_driver_s *dev;      /* Current network device */
  uint8_t lineno;                    /* Line number */
  uint8_t linesize;                  /* Number of valid characters in line[] */
  uint8_t offset;                    /* Offset to first valid character in line[] */
  uint8_t entry;                     /* See enum netprocfs_entry_e */
  char line[NET_LINELEN];            /* Pre-allocated buffer for formatted lines */
};

/* Level 1 is the directory of attributes */

struct netprocfs_level1_s
{
  struct procfs_dir_priv_s  base;    /* Base directory private data */
  char name[NAME_MAX + 1];           /* Name of last node visited */
#ifdef CONFIG_NETDEV_IFINDEX
  uint8_t ifindex;                   /* Next ifindex to visit */
#endif
};

/* Line generating function type */

typedef int (*linegen_t)(FAR struct netprocfs_file_s *netfile);

/****************************************************************************
 * Public Data
 ****************************************************************************/

#ifdef __cplusplus
#  define EXTERN extern "C"
extern "C"
{
#else
#  define EXTERN extern
#endif

/****************************************************************************
 * Public Function Prototypes
 ****************************************************************************/

/****************************************************************************
 * Name: netprocfs_read_linegen
 *
 * Description:
 *   Read and format procfs data using a line generation table.
 *
 * Input Parameters:
 *   priv   - A reference to the network procfs file structure
 *   buffer - The user-provided buffer into which device status will be
 *            returned.
 *   buflen - The size in bytes of the user provided buffer.
 *   gentab - Table of line generation functions
 *   nelems - The number of elements in the table
 *
 * Returned Value:
 *   Zero (OK) is returned on success; a negated errno value is returned
 *   on failure.
 *
 ****************************************************************************/

ssize_t netprocfs_read_linegen(FAR struct netprocfs_file_s *priv,
                                FAR char *buffer, size_t buflen,
                                FAR const linegen_t *gentab, int nelems);

/****************************************************************************
 * Name: netprocfs_read_netstats
 *
 * Description:
 *   Read and format network layer statistics.
 *
 * Input Parameters:
 *   priv - A reference to the network procfs file structure
 *   buffer - The user-provided buffer into which network status will be
 *            returned.
 *   bulen  - The size in bytes of the user provided buffer.
 *
 * Returned Value:
 *   Zero (OK) is returned on success; a negated errno value is returned
 *   on failure.
 *
 ****************************************************************************/

#ifdef CONFIG_NET_STATISTICS
ssize_t netprocfs_read_netstats(FAR struct netprocfs_file_s *priv,
                                FAR char *buffer, size_t buflen);
#endif

/****************************************************************************
 * Name: netprocfs_read_mldstats
 *
 * Description:
 *   Read and format MLD statistics.
 *
 * Input Parameters:
 *   priv - A reference to the network procfs file structure
 *   buffer - The user-provided buffer into which network status will be
 *            returned.
 *   bulen  - The size in bytes of the user provided buffer.
 *
 * Returned Value:
 *   Zero (OK) is returned on success; a negated errno value is returned
 *   on failure.
 *
 ****************************************************************************/

#if defined(CONFIG_NET_STATISTICS) && defined(CONFIG_NET_MLD)
ssize_t netprocfs_read_mldstats(FAR struct netprocfs_file_s *priv,
                                FAR char *buffer, size_t buflen);
#endif

/****************************************************************************
 * Name: netprocfs_read_routes
 *
 * Description:
 *   Read and format routing table entries.
 *
 * Input Parameters:
 *   priv - A reference to the network procfs file structure
 *   buffer - The user-provided buffer into which network status will be
 *            returned.
 *   bulen  - The size in bytes of the user provided buffer.
 *
 * Returned Value:
 *   Zero (OK) is returned on success; a negated errno value is returned
 *   on failure.
 *
 ****************************************************************************/

#ifdef CONFIG_NET_ROUTE
ssize_t netprocfs_read_routes(FAR struct netprocfs_file_s *priv,
                              FAR char *buffer, size_t buflen);
#endif

/****************************************************************************
 * Name: netprocfs_read_devstats
 *
 * Description:
 *   Read and format network device statistics.
 *
 * Input Parameters:
 *   priv - A reference to the network procfs file structure
 *   buffer - The user-provided buffer into which device status will be
 *            returned.
 *   bulen  - The size in bytes of the user provided buffer.
 *
 * Returned Value:
 *   Zero (OK) is returned on success; a negated errno value is returned
 *   on failure.
 *
 ****************************************************************************/

ssize_t netprocfs_read_devstats(FAR struct netprocfs_file_s *priv,
                                FAR char *buffer, size_t buflen);

#undef EXTERN
#ifdef __cplusplus
}
#endif

#endif /* CONFIG_FS_PROCFS && !CONFIG_FS_PROCFS_EXCLUDE_NET */
#endif /* __NET_PROCFS_PROCFS_H */
