/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.blobstore;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import org.apache.lucene.store.RateLimiter;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.metadata.SnapshotId;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.ImmutableBlobContainer;
import org.elasticsearch.common.collect.ImmutableCollection;
import org.elasticsearch.common.collect.ImmutableList;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.collect.Lists;
import org.elasticsearch.common.collect.Maps;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.compress.CompressorFactory;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.metrics.CounterMetric;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.snapshots.IndexShardRepository;
import org.elasticsearch.index.snapshots.blobstore.BlobStoreIndexShardRepository;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.RepositorySettings;
import org.elasticsearch.repositories.blobstore.BlobStoreSnapshot;
import org.elasticsearch.snapshots.InvalidSnapshotNameException;
import org.elasticsearch.snapshots.Snapshot;
import org.elasticsearch.snapshots.SnapshotCreationException;
import org.elasticsearch.snapshots.SnapshotException;
import org.elasticsearch.snapshots.SnapshotMissingException;
import org.elasticsearch.snapshots.SnapshotShardFailure;

public abstract class BlobStoreRepository
extends AbstractLifecycleComponent<Repository>
implements Repository,
BlobStoreIndexShardRepository.RateLimiterListener {
    private ImmutableBlobContainer snapshotsBlobContainer;
    protected final String repositoryName;
    private static final String SNAPSHOT_PREFIX = "snapshot-";
    private static final String SNAPSHOTS_FILE = "index";
    private static final String METADATA_PREFIX = "metadata-";
    private final BlobStoreIndexShardRepository indexShardRepository;
    private final ToXContent.Params globalOnlyFormatParams;
    private final RateLimiter snapshotRateLimiter;
    private final RateLimiter restoreRateLimiter;
    private final CounterMetric snapshotRateLimitingTimeInNanos = new CounterMetric();
    private final CounterMetric restoreRateLimitingTimeInNanos = new CounterMetric();

    protected BlobStoreRepository(String repositoryName, RepositorySettings repositorySettings, IndexShardRepository indexShardRepository) {
        super(repositorySettings.globalSettings());
        this.repositoryName = repositoryName;
        this.indexShardRepository = (BlobStoreIndexShardRepository)indexShardRepository;
        HashMap<String, String> globalOnlyParams = Maps.newHashMap();
        globalOnlyParams.put("global_persistent_only", "true");
        this.globalOnlyFormatParams = new ToXContent.MapParams(globalOnlyParams);
        this.snapshotRateLimiter = this.getRateLimiter(repositorySettings, "max_snapshot_bytes_per_sec", new ByteSizeValue(20L, ByteSizeUnit.MB));
        this.restoreRateLimiter = this.getRateLimiter(repositorySettings, "max_restore_bytes_per_sec", new ByteSizeValue(20L, ByteSizeUnit.MB));
    }

    @Override
    protected void doStart() throws ElasticsearchException {
        this.snapshotsBlobContainer = this.blobStore().immutableBlobContainer(this.basePath());
        this.indexShardRepository.initialize(this.blobStore(), this.basePath(), this.chunkSize(), this.snapshotRateLimiter, this.restoreRateLimiter, this);
    }

    @Override
    protected void doStop() throws ElasticsearchException {
    }

    @Override
    protected void doClose() throws ElasticsearchException {
        try {
            this.blobStore().close();
        }
        catch (Throwable t) {
            this.logger.warn("cannot close blob store", t, new Object[0]);
        }
    }

    protected abstract BlobStore blobStore();

    protected abstract BlobPath basePath();

    protected boolean isCompress() {
        return false;
    }

    protected ByteSizeValue chunkSize() {
        return null;
    }

    @Override
    public void initializeSnapshot(SnapshotId snapshotId, ImmutableList<String> indices, MetaData metaData) {
        try {
            BlobStoreSnapshot blobStoreSnapshot = BlobStoreSnapshot.builder().name(snapshotId.getSnapshot()).indices(indices).startTime(System.currentTimeMillis()).build();
            BytesStreamOutput bStream = this.writeSnapshot(blobStoreSnapshot);
            String snapshotBlobName = this.snapshotBlobName(snapshotId);
            if (this.snapshotsBlobContainer.blobExists(snapshotBlobName)) {
                throw new InvalidSnapshotNameException(snapshotId, "snapshot with such name already exists");
            }
            this.snapshotsBlobContainer.writeBlob(snapshotBlobName, bStream.bytes().streamInput(), bStream.bytes().length());
            bStream = this.writeGlobalMetaData(metaData);
            this.snapshotsBlobContainer.writeBlob(this.metaDataBlobName(snapshotId), bStream.bytes().streamInput(), bStream.bytes().length());
            for (String index : indices) {
                IndexMetaData indexMetaData = metaData.index(index);
                BlobPath indexPath = this.basePath().add("indices").add(index);
                ImmutableBlobContainer indexMetaDataBlobContainer = this.blobStore().immutableBlobContainer(indexPath);
                StreamOutput stream = bStream = new BytesStreamOutput();
                if (this.isCompress()) {
                    stream = CompressorFactory.defaultCompressor().streamOutput(stream);
                }
                XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON, stream);
                builder.startObject();
                IndexMetaData.Builder.toXContent(indexMetaData, builder, ToXContent.EMPTY_PARAMS);
                builder.endObject();
                builder.close();
                indexMetaDataBlobContainer.writeBlob(this.snapshotBlobName(snapshotId), bStream.bytes().streamInput(), bStream.bytes().length());
            }
        }
        catch (IOException ex) {
            throw new SnapshotCreationException(snapshotId, (Throwable)ex);
        }
    }

    @Override
    public void deleteSnapshot(SnapshotId snapshotId) {
        Snapshot snapshot = this.readSnapshot(snapshotId);
        MetaData metaData = this.readSnapshotMetaData(snapshotId, snapshot.indices());
        try {
            String blobName = this.snapshotBlobName(snapshotId);
            this.snapshotsBlobContainer.deleteBlob(blobName);
            this.snapshotsBlobContainer.deleteBlob(this.metaDataBlobName(snapshotId));
            ImmutableCollection snapshotIds = this.snapshots();
            if (snapshotIds.contains(snapshotId)) {
                ImmutableList.Builder builder = ImmutableList.builder();
                for (SnapshotId id : snapshotIds) {
                    if (snapshotId.equals(id)) continue;
                    builder.add(id);
                }
                snapshotIds = builder.build();
            }
            this.writeSnapshotList((ImmutableList<SnapshotId>)snapshotIds);
            for (String index : snapshot.indices()) {
                BlobPath indexPath = this.basePath().add("indices").add(index);
                ImmutableBlobContainer indexMetaDataBlobContainer = this.blobStore().immutableBlobContainer(indexPath);
                try {
                    indexMetaDataBlobContainer.deleteBlob(blobName);
                }
                catch (IOException ex) {
                    throw new SnapshotException(snapshotId, "failed to delete metadata", ex);
                }
                IndexMetaData indexMetaData = metaData.index(index);
                for (int i = 0; i < indexMetaData.getNumberOfShards(); ++i) {
                    this.indexShardRepository.delete(snapshotId, new ShardId(index, i));
                }
            }
        }
        catch (IOException ex) {
            throw new RepositoryException(this.repositoryName, "failed to update snapshot in repository", ex);
        }
    }

    @Override
    public Snapshot finalizeSnapshot(SnapshotId snapshotId, String failure, int totalShards, ImmutableList<SnapshotShardFailure> shardFailures) {
        BlobStoreSnapshot snapshot = (BlobStoreSnapshot)this.readSnapshot(snapshotId);
        if (snapshot == null) {
            throw new SnapshotMissingException(snapshotId);
        }
        if (snapshot.state().completed()) {
            throw new SnapshotException(snapshotId, "snapshot is already closed");
        }
        try {
            String blobName = this.snapshotBlobName(snapshotId);
            BlobStoreSnapshot.Builder updatedSnapshot = BlobStoreSnapshot.builder().snapshot(snapshot);
            if (failure == null) {
                updatedSnapshot.success();
                updatedSnapshot.failures(totalShards, shardFailures);
            } else {
                updatedSnapshot.failed(failure);
            }
            updatedSnapshot.endTime(System.currentTimeMillis());
            snapshot = updatedSnapshot.build();
            BytesStreamOutput bStream = this.writeSnapshot(snapshot);
            this.snapshotsBlobContainer.writeBlob(blobName, bStream.bytes().streamInput(), bStream.bytes().length());
            ImmutableCollection snapshotIds = this.snapshots();
            if (!snapshotIds.contains(snapshotId)) {
                snapshotIds = ((ImmutableList.Builder)((ImmutableList.Builder)ImmutableList.builder().addAll((Iterable)snapshotIds)).add(snapshotId)).build();
            }
            this.writeSnapshotList((ImmutableList<SnapshotId>)snapshotIds);
            return snapshot;
        }
        catch (IOException ex) {
            throw new RepositoryException(this.repositoryName, "failed to update snapshot in repository", ex);
        }
    }

    @Override
    public ImmutableList<SnapshotId> snapshots() {
        try {
            ImmutableMap<String, BlobMetaData> blobs;
            ArrayList<SnapshotId> snapshots = Lists.newArrayList();
            try {
                blobs = this.snapshotsBlobContainer.listBlobsByPrefix(SNAPSHOT_PREFIX);
            }
            catch (UnsupportedOperationException ex) {
                return this.readSnapshotList();
            }
            int prefixLength = SNAPSHOT_PREFIX.length();
            for (BlobMetaData md : blobs.values()) {
                String name = md.name().substring(prefixLength);
                snapshots.add(new SnapshotId(this.repositoryName, name));
            }
            return ImmutableList.copyOf(snapshots);
        }
        catch (IOException ex) {
            throw new RepositoryException(this.repositoryName, "failed to list snapshots in repository", ex);
        }
    }

    @Override
    public MetaData readSnapshotMetaData(SnapshotId snapshotId, ImmutableList<String> indices) {
        MetaData metaData;
        try {
            byte[] data = this.snapshotsBlobContainer.readBlobFully(this.metaDataBlobName(snapshotId));
            metaData = this.readMetaData(data);
        }
        catch (FileNotFoundException ex) {
            throw new SnapshotMissingException(snapshotId, (Throwable)ex);
        }
        catch (IOException ex) {
            throw new SnapshotException(snapshotId, "failed to get snapshots", ex);
        }
        MetaData.Builder metaDataBuilder = MetaData.builder(metaData);
        for (String index : indices) {
            BlobPath indexPath = this.basePath().add("indices").add(index);
            ImmutableBlobContainer indexMetaDataBlobContainer = this.blobStore().immutableBlobContainer(indexPath);
            XContentParser parser = null;
            try {
                byte[] data = indexMetaDataBlobContainer.readBlobFully(this.snapshotBlobName(snapshotId));
                parser = XContentHelper.createParser(data, 0, data.length);
                XContentParser.Token token = parser.nextToken();
                if (token == XContentParser.Token.START_OBJECT) {
                    IndexMetaData indexMetaData = IndexMetaData.Builder.fromXContent(parser);
                    token = parser.nextToken();
                    if (token == XContentParser.Token.END_OBJECT) {
                        metaDataBuilder.put(indexMetaData, false);
                        continue;
                    }
                }
                try {
                    throw new ElasticsearchParseException("unexpected token  [" + (Object)((Object)token) + "]");
                }
                catch (IOException ex) {
                    throw new SnapshotException(snapshotId, "failed to read metadata", ex);
                }
            }
            finally {
                if (parser == null) continue;
                parser.close();
            }
        }
        return metaDataBuilder.build();
    }

    @Override
    public Snapshot readSnapshot(SnapshotId snapshotId) {
        try {
            String blobName = this.snapshotBlobName(snapshotId);
            byte[] data = this.snapshotsBlobContainer.readBlobFully(blobName);
            return this.readSnapshot(data);
        }
        catch (FileNotFoundException ex) {
            throw new SnapshotMissingException(snapshotId, (Throwable)ex);
        }
        catch (IOException ex) {
            throw new SnapshotException(snapshotId, "failed to get snapshots", ex);
        }
    }

    private RateLimiter getRateLimiter(RepositorySettings repositorySettings, String setting, ByteSizeValue defaultRate) {
        ByteSizeValue maxSnapshotBytesPerSec = repositorySettings.settings().getAsBytesSize(setting, this.componentSettings.getAsBytesSize(setting, defaultRate));
        if (maxSnapshotBytesPerSec.bytes() <= 0L) {
            return null;
        }
        return new RateLimiter.SimpleRateLimiter(maxSnapshotBytesPerSec.mbFrac());
    }

    private BlobStoreSnapshot readSnapshot(byte[] data) throws IOException {
        XContentParser parser = null;
        try {
            parser = XContentHelper.createParser(data, 0, data.length);
            XContentParser.Token token = parser.nextToken();
            if (token == XContentParser.Token.START_OBJECT && (token = parser.nextToken()) == XContentParser.Token.FIELD_NAME) {
                parser.nextToken();
                BlobStoreSnapshot snapshot = BlobStoreSnapshot.Builder.fromXContent(parser);
                token = parser.nextToken();
                if (token == XContentParser.Token.END_OBJECT) {
                    BlobStoreSnapshot blobStoreSnapshot = snapshot;
                    return blobStoreSnapshot;
                }
            }
            throw new ElasticsearchParseException("unexpected token  [" + (Object)((Object)token) + "]");
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
    }

    private MetaData readMetaData(byte[] data) throws IOException {
        XContentParser parser = null;
        try {
            parser = XContentHelper.createParser(data, 0, data.length);
            XContentParser.Token token = parser.nextToken();
            if (token == XContentParser.Token.START_OBJECT && (token = parser.nextToken()) == XContentParser.Token.FIELD_NAME) {
                parser.nextToken();
                MetaData metaData = MetaData.Builder.fromXContent(parser);
                token = parser.nextToken();
                if (token == XContentParser.Token.END_OBJECT) {
                    MetaData metaData2 = metaData;
                    return metaData2;
                }
            }
            throw new ElasticsearchParseException("unexpected token  [" + (Object)((Object)token) + "]");
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
    }

    private String snapshotBlobName(SnapshotId snapshotId) {
        return SNAPSHOT_PREFIX + snapshotId.getSnapshot();
    }

    private String metaDataBlobName(SnapshotId snapshotId) {
        return METADATA_PREFIX + snapshotId.getSnapshot();
    }

    private BytesStreamOutput writeSnapshot(BlobStoreSnapshot snapshot) throws IOException {
        BytesStreamOutput bStream;
        StreamOutput stream = bStream = new BytesStreamOutput();
        if (this.isCompress()) {
            stream = CompressorFactory.defaultCompressor().streamOutput(stream);
        }
        XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON, stream);
        builder.startObject();
        BlobStoreSnapshot.Builder.toXContent(snapshot, builder, this.globalOnlyFormatParams);
        builder.endObject();
        builder.close();
        return bStream;
    }

    private BytesStreamOutput writeGlobalMetaData(MetaData metaData) throws IOException {
        BytesStreamOutput bStream;
        StreamOutput stream = bStream = new BytesStreamOutput();
        if (this.isCompress()) {
            stream = CompressorFactory.defaultCompressor().streamOutput(stream);
        }
        XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON, stream);
        builder.startObject();
        MetaData.Builder.toXContent(metaData, builder, this.globalOnlyFormatParams);
        builder.endObject();
        builder.close();
        return bStream;
    }

    protected void writeSnapshotList(ImmutableList<SnapshotId> snapshots) throws IOException {
        BytesStreamOutput bStream;
        StreamOutput stream = bStream = new BytesStreamOutput();
        if (this.isCompress()) {
            stream = CompressorFactory.defaultCompressor().streamOutput(stream);
        }
        XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON, stream);
        builder.startObject();
        builder.startArray("snapshots");
        for (SnapshotId snapshot : snapshots) {
            builder.value(snapshot.getSnapshot());
        }
        builder.endArray();
        builder.endObject();
        builder.close();
        this.snapshotsBlobContainer.writeBlob(SNAPSHOTS_FILE, bStream.bytes().streamInput(), bStream.bytes().length());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected ImmutableList<SnapshotId> readSnapshotList() throws IOException {
        byte[] data = this.snapshotsBlobContainer.readBlobFully(SNAPSHOTS_FILE);
        ArrayList<SnapshotId> snapshots = new ArrayList<SnapshotId>();
        XContentParser parser = null;
        try {
            String currentFieldName;
            parser = XContentHelper.createParser(data, 0, data.length);
            if (parser.nextToken() == XContentParser.Token.START_OBJECT && parser.nextToken() == XContentParser.Token.FIELD_NAME && "snapshots".equals(currentFieldName = parser.currentName()) && parser.nextToken() == XContentParser.Token.START_ARRAY) {
                while (parser.nextToken() != XContentParser.Token.END_ARRAY) {
                    snapshots.add(new SnapshotId(this.repositoryName, parser.text()));
                }
            }
        }
        finally {
            if (parser != null) {
                parser.close();
            }
        }
        return ImmutableList.copyOf(snapshots);
    }

    @Override
    public void onRestorePause(long nanos) {
        this.restoreRateLimitingTimeInNanos.inc(nanos);
    }

    @Override
    public void onSnapshotPause(long nanos) {
        this.snapshotRateLimitingTimeInNanos.inc(nanos);
    }

    @Override
    public long snapshotThrottleTimeInNanos() {
        return this.snapshotRateLimitingTimeInNanos.count();
    }

    @Override
    public long restoreThrottleTimeInNanos() {
        return this.restoreRateLimitingTimeInNanos.count();
    }
}

