/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.util.time;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.base.Supplier;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.Locale;
import java.util.SimpleTimeZone;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import org.apache.brooklyn.util.collections.MutableList;
import org.apache.brooklyn.util.exceptions.Exceptions;
import org.apache.brooklyn.util.guava.Maybe;
import org.apache.brooklyn.util.text.Strings;
import org.apache.brooklyn.util.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Time {
    private static final Logger log = LoggerFactory.getLogger(Time.class);
    public static final String DATE_FORMAT_PREFERRED_W_TZ = "yyyy-MM-dd HH:mm:ss.SSS Z";
    public static final String DATE_FORMAT_PREFERRED = "yyyy-MM-dd HH:mm:ss.SSS";
    public static final String DATE_FORMAT_STAMP = "yyyyMMdd-HHmmssSSS";
    public static final String DATE_FORMAT_SIMPLE_STAMP = "yyyy-MM-dd-HHmm";
    public static final String DATE_FORMAT_OF_DATE_TOSTRING = "EEE MMM dd HH:mm:ss zzz yyyy";
    public static final String DATE_FORMAT_ISO8601 = "yyyy-MM-dd'T'HH:mm:ss.SSSZ";
    public static final String DATE_FORMAT_ISO8601_NO_MILLIS = "yyyy-MM-dd'T'HH:mm:ssZ";
    public static final long MILLIS_IN_SECOND = 1000L;
    public static final long MILLIS_IN_MINUTE = 60000L;
    public static final long MILLIS_IN_HOUR = 3600000L;
    public static final long MILLIS_IN_DAY = 86400000L;
    public static final long MILLIS_IN_YEAR = 31536000000L;
    public static final TimeZone TIME_ZONE_UTC = TimeZone.getTimeZone("");
    private static Function<Long, String> dateString = new Function<Long, String>(){

        @Nullable
        public String apply(@Nullable Long input) {
            if (input == null) {
                return null;
            }
            return Time.makeDateString(input);
        }
    };
    private static Function<Long, String> dateStampString = new Function<Long, String>(){

        @Nullable
        public String apply(@Nullable Long input) {
            if (input == null) {
                return null;
            }
            return Time.makeDateStampString(input);
        }
    };
    private static final Function<Long, String> LONG_TO_TIME_STRING_EXACT = new FunctionLongToTimeStringExact();
    @Deprecated
    private static Function<Long, String> timeString = new Function<Long, String>(){

        @Nullable
        public String apply(@Nullable Long input) {
            if (input == null) {
                return null;
            }
            return Time.makeTimeStringExact(input);
        }
    };
    private static final Function<Long, String> LONG_TO_TIME_STRING_ROUNDED = new FunctionLongToTimeStringRounded();
    @Deprecated
    private static Function<Long, String> timeStringRounded = new Function<Long, String>(){

        @Nullable
        public String apply(@Nullable Long input) {
            if (input == null) {
                return null;
            }
            return Time.makeTimeStringRounded(input);
        }
    };
    private static final Function<Duration, String> DURATION_TO_TIME_STRING_ROUNDED = new FunctionDurationToTimeStringRounded();
    private static final String DIGIT = "\\d";
    private static final String LETTER = "\\p{L}";
    private static final String COMMON_SEPARATORS = "-\\.";
    private static final String TIME_SEPARATOR = "-\\.:";
    private static final String DATE_SEPARATOR = "-\\./ ";
    private static final String DATE_TIME_ANY_ORDER_GROUP_SEPARATOR = "-\\.:/ ";
    private static final String DATE_ONLY_WITH_INNER_SEPARATORS = Time.namedGroup("year", "\\d\\d\\d\\d") + Time.anyChar("-\\./ ") + Time.namedGroup("month", Time.options(Time.optionally("\\d") + "\\d", Time.anyChar("\\p{L}") + "+")) + Time.anyChar("-\\./ ") + Time.namedGroup("day", Time.optionally("\\d") + "\\d");
    private static final String DATE_WORDS_2 = Time.namedGroup("month", Time.anyChar("\\p{L}") + "+") + Time.anyChar("-\\./ ") + Time.namedGroup("day", Time.optionally("\\d") + "\\d") + ",?" + Time.anyChar("-\\./ ") + "+" + Time.namedGroup("year", "\\d\\d\\d\\d");
    private static final String DATE_WORDS_3 = Time.namedGroup("day", Time.optionally("\\d") + "\\d") + Time.anyChar("-\\./ ") + Time.namedGroup("month", Time.anyChar("\\p{L}") + "+") + ",?" + Time.anyChar("-\\./ ") + "+" + Time.namedGroup("year", "\\d\\d\\d\\d");
    private static final String DATE_ONLY_NO_SEPARATORS = Time.namedGroup("year", "\\d\\d\\d\\d") + Time.namedGroup("month", "\\d\\d") + Time.namedGroup("day", "\\d\\d");
    private static final String MERIDIAN = Time.anyChar("aApP") + Time.optionally(Time.anyChar("mM"));
    private static final String TIME_ONLY_WITH_INNER_SEPARATORS = Time.namedGroup("hours", Time.optionally("\\d") + "\\d") + Time.optionally(Time.anyChar("-\\.:") + Time.namedGroup("mins", "\\d\\d") + Time.optionally(Time.anyChar("-\\.:") + Time.namedGroup("secs", "\\d\\d" + Time.optionally(Time.optionally("\\.") + "\\d" + "+")))) + Time.optionally(" *" + Time.namedGroup("meridian", Time.notMatching("\\p{L}\\p{L}\\p{L}") + MERIDIAN));
    private static final String TIME_ONLY_NO_SEPARATORS = Time.namedGroup("hours", "\\d\\d") + Time.namedGroup("mins", "\\d\\d") + Time.optionally(Time.namedGroup("secs", "\\d\\d" + Time.optionally(Time.optionally("\\.") + "\\d" + "+"))) + Time.namedGroup("meridian", "");
    private static final String TZ_CODE = Time.namedGroup("tzCode", Time.notMatching(MERIDIAN + Time.options("$", Time.anyChar("^\\p{L}"))) + Time.anyChar("\\p{L}") + "+" + Time.anyChar("\\p{L}\\d\\/\\-\\' _") + "*");
    private static final String TIME_ZONE_SIGNED_OFFSET = Time.namedGroup("tz", Time.options(Time.namedGroup("tzOffset", Time.options("\\+", "-") + "\\d" + Time.optionally("\\d") + Time.optionally(Time.optionally(":") + "\\d" + "\\d")), Time.optionally("\\+") + TZ_CODE));
    private static final String TIME_ZONE_OPTIONALLY_SIGNED_OFFSET = Time.namedGroup("tz", Time.options(Time.namedGroup("tzOffset", Time.options("\\+", "-", " ") + Time.options("0\\d", "10", "11", "12") + Time.optionally(Time.optionally(":") + "\\d" + "\\d")), TZ_CODE));

    public static String makeDateString() {
        return Time.makeDateString(System.currentTimeMillis());
    }

    public static String makeDateString(long date) {
        return Time.makeDateString(new Date(date), DATE_FORMAT_PREFERRED);
    }

    public static String makeDateString(Date date) {
        return Time.makeDateString(date, DATE_FORMAT_PREFERRED);
    }

    public static String makeDateString(Date date, @Nullable String format) {
        return Time.makeDateString(date, format, null);
    }

    public static String makeDateStringUtc(Date date, String format) {
        return Time.makeDateString(date, format, TimeZone.getTimeZone("UTC"));
    }

    public static String makeDateStringUtc(Date date) {
        return Time.makeDateString(date, TimeZone.getTimeZone("UTC"));
    }

    public static String makeDateString(Date date, @Nullable TimeZone tz) {
        return Time.makeDateString(date, null, null);
    }

    public static String makeDateString(Date date, @Nullable String format, @Nullable TimeZone tz) {
        SimpleDateFormat fmt = new SimpleDateFormat(format != null ? format : (tz == null ? DATE_FORMAT_PREFERRED : DATE_FORMAT_PREFERRED_W_TZ));
        if (tz != null) {
            fmt.setTimeZone(tz);
        }
        return fmt.format(date);
    }

    public static String makeDateString(Calendar date) {
        return Time.makeDateString(date.getTime(), DATE_FORMAT_PREFERRED_W_TZ);
    }

    public static String makeDateString(Calendar date, String format) {
        return Time.makeDateString(date.getTime(), format, date.getTimeZone());
    }

    public static String makeDateString(Instant date, String format, Locale locale, ZoneId zone) {
        DateTimeFormatter fmt = DateTimeFormatter.ofPattern(format).withLocale(locale == null ? Locale.ROOT : locale).withZone(zone == null ? ZoneId.of("UTC") : zone);
        return fmt.format(date);
    }

    public static String makeDateString(Instant date) {
        String result = Time.makeDateString(date, DATE_FORMAT_PREFERRED, null, null);
        result = Strings.removeFromEnd(result, ".000");
        result = Strings.removeFromEnd(result, ":00");
        return result;
    }

    public static Function<Long, String> toDateString() {
        return dateString;
    }

    public static String makeDateStampString() {
        return Time.makeDateStampString(System.currentTimeMillis());
    }

    public static String makeDateStampStringZ() {
        return Time.makeDateStampString(System.currentTimeMillis()) + "Z";
    }

    public static String makeDateStampStringZ(Instant instant) {
        String s = Time.makeDateStampStringZ(instant.toEpochMilli());
        if (s.endsWith("000")) {
            s = Strings.removeFromEnd(s, "000");
            s = Strings.removeFromEnd(s, "00");
        }
        return s + "Z";
    }

    public static String makeDateStampStringZ(Date date) {
        String s = Time.makeDateStampStringZ(date.getTime());
        if (s.endsWith("000")) {
            s = Strings.removeFromEnd(s, "000");
            s = Strings.removeFromEnd(s, "00");
        }
        return s + "Z";
    }

    public static String makeIso8601DateString() {
        return Time.replaceZeroZoneWithZ(Time.makeIso8601DateStringLocal(Instant.now()));
    }

    public static String makeIso8601DateStringZ(Instant instant) {
        return Time.replaceZeroZoneWithZ(Time.makeDateString(instant, DATE_FORMAT_ISO8601, null, null));
    }

    private static String replaceZeroZoneWithZ(String s) {
        if (s == null) {
            return s;
        }
        String sz = null;
        if (s.endsWith("+0000")) {
            sz = Strings.removeFromEnd(s, "+0000");
        } else if (s.endsWith("+00:00")) {
            sz = Strings.removeFromEnd(s, "+00:00");
        }
        if (sz == null) {
            return s;
        }
        return sz + "Z";
    }

    public static String makeIso8601DateStringLocal(Instant instant) {
        return Time.makeDateString(instant, DATE_FORMAT_ISO8601, Locale.getDefault(), ZoneId.systemDefault());
    }

    public static String makeIso8601DateString(Date date) {
        return Time.replaceZeroZoneWithZ(Time.makeDateStringUtc(date, DATE_FORMAT_ISO8601));
    }

    public static String makeDateStampString(long date) {
        return new SimpleDateFormat(DATE_FORMAT_STAMP).format(new Date(date));
    }

    public static String makeDateStampStringZ(long date) {
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT_STAMP);
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        return dateFormat.format(new Date(date));
    }

    public static String makeDateSimpleStampString() {
        return Time.makeDateSimpleStampString(System.currentTimeMillis());
    }

    public static String makeDateSimpleStampString(long date) {
        return new SimpleDateFormat(DATE_FORMAT_SIMPLE_STAMP).format(new Date(date));
    }

    public static Function<Long, String> toDateStampString() {
        return dateStampString;
    }

    public static String makeTimeStringExact(long t, TimeUnit unit) {
        long nanos = unit.toNanos(t);
        return Time.makeTimeStringNanoExact(nanos);
    }

    public static String makeTimeStringRounded(long t, TimeUnit unit) {
        long nanos = unit.toNanos(t);
        return Time.makeTimeStringNanoRounded(nanos);
    }

    public static String makeTimeStringRounded(@Nullable Stopwatch timer) {
        return timer == null ? null : Time.makeTimeStringRounded(timer.elapsed(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
    }

    public static String makeTimeStringExact(long t) {
        return Time.makeTimeString(t, false);
    }

    public static String makeTimeStringRounded(long t) {
        return Time.makeTimeString(t, true);
    }

    public static String makeTimeStringRoundedSince(long utc) {
        return Time.makeTimeString(System.currentTimeMillis() - utc, true);
    }

    public static String makeTimeStringExact(@Nullable Duration d) {
        return d == null ? null : Time.makeTimeStringNanoExact(d.toNanoseconds());
    }

    public static String makeTimeStringRounded(@Nullable Duration d) {
        return d == null ? null : Time.makeTimeStringNanoRounded(d.toNanoseconds());
    }

    public static String makeTimeString(long t, boolean round) {
        return Time.makeTimeStringNano(t * 1000000L, round);
    }

    public static String makeTimeStringNanoExact(long tn) {
        return Time.makeTimeStringNano(tn, false);
    }

    public static String makeTimeStringNanoRounded(long tn) {
        return Time.makeTimeStringNano(tn, true);
    }

    public static String makeTimeStringNano(long tn, boolean round) {
        if (tn < 0L) {
            return "-" + Time.makeTimeStringNano(-tn, round);
        }
        if (tn == 0L) {
            return "0ms";
        }
        long tnm = tn % 1000000L;
        long t = tn / 1000000L;
        String result = "";
        long d = t / 86400000L;
        long h = (t %= 86400000L) / 3600000L;
        long m = (t %= 3600000L) / 60000L;
        long s = (t %= 60000L) / 1000L;
        long ms = t %= 1000L;
        int segments = 0;
        if (d > 0L) {
            result = result + d + "d ";
            ++segments;
        }
        if (h > 0L) {
            result = result + h + "h ";
            ++segments;
        }
        if (round && segments >= 2) {
            return Strings.removeAllFromEnd(result, " ");
        }
        if (m > 0L) {
            result = result + m + "m ";
            ++segments;
        }
        if (round && (segments >= 2 || d > 0L)) {
            return Strings.removeAllFromEnd(result, " ");
        }
        if (s > 0L) {
            if (ms == 0L && tnm == 0L) {
                result = result + s + "s";
                ++segments;
                return result;
            }
            if (round && segments > 0) {
                result = result + s + "s";
                ++segments;
                return result;
            }
            if (round && s > 10L) {
                result = result + Time.toDecimal(s, (double)ms / 1000.0, 1) + "s";
                ++segments;
                return result;
            }
            if (round) {
                result = result + Time.toDecimal(s, (double)ms / 1000.0, 2) + "s";
                ++segments;
                return result;
            }
            result = result + s + "s ";
        }
        if (round && segments > 0) {
            return Strings.removeAllFromEnd(result, " ");
        }
        if (ms > 0L) {
            if (tnm == 0L) {
                result = result + ms + "ms";
                ++segments;
                return result;
            }
            if (round && ms >= 100L) {
                result = result + Time.toDecimal(ms, (double)tnm / 1000000.0, 1) + "ms";
                ++segments;
                return result;
            }
            if (round && ms >= 10L) {
                result = result + Time.toDecimal(ms, (double)tnm / 1000000.0, 2) + "ms";
                ++segments;
                return result;
            }
            if (round) {
                result = result + Time.toDecimal(ms, (double)tnm / 1000000.0, 3) + "ms";
                ++segments;
                return result;
            }
            result = result + ms + "ms ";
        }
        long us = tnm / 1000L;
        long ns = tnm % 1000L;
        if (us > 0L) {
            if (ns == 0L) {
                result = result + us + "us";
                ++segments;
                return result;
            }
            if (round && us >= 100L) {
                result = result + Time.toDecimal(us, (double)ns / 1000.0, 1) + "us";
                ++segments;
                return result;
            }
            if (round && us >= 10L) {
                result = result + Time.toDecimal(us, (double)ns / 1000.0, 2) + "us";
                ++segments;
                return result;
            }
            if (round) {
                result = result + Time.toDecimal(us, (double)ns / 1000.0, 3) + "us";
                ++segments;
                return result;
            }
            result = result + us + "us ";
        }
        if (ns > 0L) {
            result = result + ns + "ns";
        }
        return Strings.removeAllFromEnd(result, " ");
    }

    public static Function<Long, String> fromLongToTimeStringExact() {
        return LONG_TO_TIME_STRING_EXACT;
    }

    public static Function<Long, String> fromLongToTimeStringRounded() {
        return LONG_TO_TIME_STRING_ROUNDED;
    }

    public static Function<Duration, String> fromDurationToTimeStringRounded() {
        return DURATION_TO_TIME_STRING_ROUNDED;
    }

    private static String toDecimal(long intPart, double fracPart, int decimalPrecision) {
        long powTen = 1L;
        for (int i = 0; i < decimalPrecision; ++i) {
            powTen *= 10L;
        }
        long fpr = Math.round(fracPart * (double)powTen);
        if (fpr == powTen) {
            ++intPart;
            fpr = 0L;
        }
        return intPart + "." + Strings.makePaddedString("" + fpr, decimalPrecision, "0", "");
    }

    public static void sleep(long millis) {
        try {
            if (millis > 0L) {
                Thread.sleep(millis);
            }
        }
        catch (InterruptedException e) {
            throw Exceptions.propagate(e);
        }
    }

    public static void sleep(Duration duration) {
        Time.sleep(duration.toMillisecondsRoundingUp());
    }

    public static long getTimeOfDayFromUtc(long timeUtc) {
        GregorianCalendar gregorianCalendar = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
        gregorianCalendar.setTimeInMillis(timeUtc);
        int hour = gregorianCalendar.get(11);
        int min = gregorianCalendar.get(12);
        int sec = gregorianCalendar.get(13);
        int millis = gregorianCalendar.get(14);
        return ((hour * 60 + min) * 60 + sec) * 1000 + millis;
    }

    public static long getTimeUtc(TimeZone zone, int year, int month, int date, int hourOfDay, int minute, int second, int millis) {
        GregorianCalendar time = new GregorianCalendar(zone);
        time.set(year, month, date, hourOfDay, minute, second);
        time.set(14, millis);
        return time.getTimeInMillis();
    }

    public static long roundFromMillis(long millis, TimeUnit units) {
        if (units.compareTo(TimeUnit.MILLISECONDS) > 0) {
            double result = (double)millis / (double)units.toMillis(1L);
            return Math.round(result);
        }
        return units.convert(millis, TimeUnit.MILLISECONDS);
    }

    public static long roundFromMillis(long millis, long millisPerUnit) {
        double result = (double)millis / (double)millisPerUnit;
        return Math.round(result);
    }

    public static long timeRemaining(long startTime, long maxTime) {
        if (maxTime == 0L) {
            return 0L;
        }
        long result = startTime + maxTime - System.currentTimeMillis();
        return result == 0L ? -1L : result;
    }

    public static Duration parseDuration(@Nullable String timeString) {
        return Duration.parse(timeString);
    }

    public static long parseElapsedTime(String timeString) {
        return (long)Time.parseElapsedTimeAsDouble(timeString);
    }

    public static double parseElapsedTimeAsDouble(String timeString) {
        return Time.parseElapsedTimeAsDouble(timeString, timeString, true, true);
    }

    private static double parseElapsedTimeAsDouble(String timeStringOrig, String timeString, boolean allowNonUnit, boolean allowNegative) {
        if (timeString == null) {
            throw new NullPointerException("GeneralHelper.parseTimeString cannot parse a null string");
        }
        try {
            if (allowNonUnit && !timeString.trim().matches(".*[A-Za-z]$")) {
                double d = Double.parseDouble(timeString);
                return d;
            }
        }
        catch (NumberFormatException e) {
            log.trace("Unable to parse '%s' as pure number. Trying smart parse.", (Object)timeStringOrig, (Object)e);
        }
        boolean isNegative = false;
        String multipleUnitsDisallowedBecause = null;
        try {
            char c;
            timeString = timeString.trim();
            int i = 0;
            if (timeString.startsWith("-")) {
                if (!allowNegative) {
                    throw new NumberFormatException("Negation is not permitted on an individual time unit in a duration");
                }
                isNegative = true;
                timeString = timeString.substring(1);
                if (!timeString.startsWith(" ")) {
                    multipleUnitsDisallowedBecause = "Negation must have a space after it to apply to mulitple units";
                } else {
                    while (timeString.startsWith(" ")) {
                        timeString = timeString.substring(1);
                    }
                }
            }
            while (i < timeString.length() && ((c = timeString.charAt(i)) == '.' || Character.isDigit(c))) {
                ++i;
            }
            String num = timeString.substring(0, i);
            timeString = timeString.substring(i).trim();
            long multiplier = 0L;
            if (num.isEmpty()) {
                if (allowNegative && !isNegative && (timeString.equalsIgnoreCase("never") || timeString.equalsIgnoreCase("off") || timeString.equalsIgnoreCase("false"))) {
                    return -1.0;
                }
                throw new NumberFormatException("Invalid duration string '" + timeStringOrig + "'");
            }
            String s = Strings.getFirstWord(timeString);
            timeString = timeString.substring(s.length()).trim();
            if (s.equalsIgnoreCase("ms") || s.equalsIgnoreCase("milli") || s.equalsIgnoreCase("millis") || s.equalsIgnoreCase("millisec") || s.equalsIgnoreCase("millisecs") || s.equalsIgnoreCase("millisecond") || s.equalsIgnoreCase("milliseconds")) {
                multiplier = 1L;
            } else if (s.equalsIgnoreCase("s") || s.equalsIgnoreCase("sec") || s.equalsIgnoreCase("secs") || s.equalsIgnoreCase("second") || s.equalsIgnoreCase("seconds")) {
                multiplier = 1000L;
            } else if (s.equalsIgnoreCase("m") || s.equalsIgnoreCase("min") || s.equalsIgnoreCase("mins") || s.equalsIgnoreCase("minute") || s.equalsIgnoreCase("minutes")) {
                multiplier = 60000L;
            } else if (s.equalsIgnoreCase("h") || s.equalsIgnoreCase("hr") || s.equalsIgnoreCase("hrs") || s.equalsIgnoreCase("hour") || s.equalsIgnoreCase("hours")) {
                multiplier = 3600000L;
            } else if (s.equalsIgnoreCase("d") || s.equalsIgnoreCase("day") || s.equalsIgnoreCase("days")) {
                multiplier = 86400000L;
            } else {
                throw new NumberFormatException("Unknown unit '" + s + "' in time string '" + timeStringOrig + "'");
            }
            double d = Double.parseDouble(num);
            double dd = 0.0;
            if (timeString.length() > 0) {
                if (multipleUnitsDisallowedBecause != null) {
                    throw new NumberFormatException(multipleUnitsDisallowedBecause);
                }
                dd = Time.parseElapsedTimeAsDouble(timeStringOrig, timeString, false, false);
                if (dd == -1.0) {
                    throw new NumberFormatException("Cannot combine '" + timeString + "' with '" + num + " " + s + "'");
                }
            }
            return (d * (double)multiplier + dd) * (double)(isNegative ? -1 : 1);
        }
        catch (Exception ex) {
            if (ex instanceof NumberFormatException) {
                throw (NumberFormatException)ex;
            }
            log.trace("Details of parse failure:", (Throwable)ex);
            throw new NumberFormatException("Cannot parse time string '" + timeStringOrig + "'");
        }
    }

    public static Calendar newCalendarFromMillisSinceEpochUtc(long timestamp) {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(timestamp);
        return cal;
    }

    public static Calendar newCalendarFromDate(Date date) {
        return Time.newCalendarFromMillisSinceEpochUtc(date.getTime());
    }

    public static Date parseDate(@Nullable String input) {
        if (input == null) {
            return null;
        }
        return Time.parseCalendarMaybe(input).get().getTime();
    }

    public static Instant parseInstant(@Nullable String input) {
        return Time.parseCalendarMaybe(input).get().toInstant();
    }

    public static Calendar parseCalendar(@Nullable String input) {
        if (input == null) {
            return null;
        }
        return Time.parseCalendarMaybe(input).get();
    }

    public static Maybe<Calendar> parseCalendarMaybe(@Nullable String input) {
        if (input == null) {
            return Maybe.absent("value is null");
        }
        if ((input = input.trim()).equalsIgnoreCase("now")) {
            return Maybe.of(Calendar.getInstance());
        }
        Maybe<Calendar> result = Time.parseCalendarUtc(input);
        if (result.isPresent()) {
            return result;
        }
        result = Time.parseCalendarSimpleFlexibleFormatParser(input);
        if (result.isPresent()) {
            return result;
        }
        Maybe<Calendar> returnResult = result;
        result = Time.parseCalendarFormat(input, new SimpleDateFormat(DATE_FORMAT_OF_DATE_TOSTRING, Locale.ROOT));
        if (result.isPresent()) {
            return result;
        }
        result = Time.parseCalendarDefaultParse(input);
        if (result.isPresent()) {
            return result;
        }
        return returnResult;
    }

    private static Maybe<Calendar> parseCalendarDefaultParse(String input) {
        try {
            long ms = Date.parse(input);
            if (ms >= new Date(1999, 12, 25).getTime() && ms <= new Date(2200, 1, 2).getTime()) {
                GregorianCalendar c = new GregorianCalendar();
                c.setTimeInMillis(ms);
                return Maybe.of(c);
            }
        }
        catch (Exception e) {
            Exceptions.propagateIfFatal(e);
        }
        return Maybe.absent();
    }

    private static Maybe<Calendar> parseCalendarUtc(String input) {
        if ((input = input.trim()).matches("\\d+")) {
            if ("0".equals(input)) {
                return Maybe.of(Time.newCalendarFromMillisSinceEpochUtc(0L));
            }
            Maybe<Calendar> result = Maybe.of(Time.newCalendarFromMillisSinceEpochUtc(Long.parseLong(input)));
            if (result.isPresent()) {
                int year = result.get().get(1);
                if (year >= 2000 && year < 2200) {
                    return result;
                }
                return Maybe.absent("long is probably not millis since epoch UTC; millis as string is not in acceptable range");
            }
        }
        return Maybe.absent("not long millis since epoch UTC");
    }

    private static String getDateTimeSeparatorPattern(String extraChars) {
        return Time.options(" +" + Time.optionally(Time.anyChar(DATE_TIME_ANY_ORDER_GROUP_SEPARATOR + extraChars + ",")), Time.anyChar(DATE_TIME_ANY_ORDER_GROUP_SEPARATOR + extraChars + ",")) + Time.anyChar(DATE_TIME_ANY_ORDER_GROUP_SEPARATOR + extraChars) + "*";
    }

    private static Maybe<Calendar> parseCalendarSimpleFlexibleFormatParser(String input) {
        String p;
        input = input.trim();
        String[] DATE_PATTERNS = new String[]{DATE_ONLY_WITH_INNER_SEPARATORS, DATE_ONLY_NO_SEPARATORS, DATE_WORDS_2, DATE_WORDS_3};
        String[] TIME_PATTERNS = new String[]{TIME_ONLY_WITH_INNER_SEPARATORS, TIME_ONLY_NO_SEPARATORS};
        String[] TZ_PATTERNS = new String[]{Time.optionally(Time.getDateTimeSeparatorPattern("")) + " " + TIME_ZONE_SIGNED_OFFSET, Time.namedGroup("tz", Time.namedGroup("tzOffset", "") + Time.namedGroup("tzCode", "")), Time.getDateTimeSeparatorPattern("") + TIME_ZONE_SIGNED_OFFSET, Time.optionally(Time.getDateTimeSeparatorPattern("")) + TIME_ZONE_OPTIONALLY_SIGNED_OFFSET};
        MutableList basePatterns = MutableList.of();
        String[] DATE_PATTERNS_UNCLOSED = new String[]{DATE_ONLY_WITH_INNER_SEPARATORS + "(" + Time.getDateTimeSeparatorPattern("Tt"), DATE_ONLY_NO_SEPARATORS + "(" + Time.optionally(Time.getDateTimeSeparatorPattern("Tt")), DATE_WORDS_2 + "(" + Time.getDateTimeSeparatorPattern("Tt"), DATE_WORDS_3 + "(" + Time.getDateTimeSeparatorPattern("Tt")};
        for (String tzP : TZ_PATTERNS) {
            for (String dateP : DATE_PATTERNS_UNCLOSED) {
                for (String timeP : TIME_PATTERNS) {
                    basePatterns.add(dateP + timeP + ")?" + tzP);
                }
            }
        }
        for (String tzP : TZ_PATTERNS) {
            for (String dateP : DATE_PATTERNS) {
                for (String timeP : TIME_PATTERNS) {
                    basePatterns.add(timeP + Time.getDateTimeSeparatorPattern("") + dateP + tzP);
                }
            }
        }
        for (String tzP : TZ_PATTERNS) {
            for (String dateP : DATE_PATTERNS) {
                for (String timeP : TIME_PATTERNS) {
                    basePatterns.add(timeP + tzP + Time.getDateTimeSeparatorPattern("") + dateP);
                }
            }
        }
        Maybe<Object> mm = Maybe.absent();
        Iterator iterator = basePatterns.iterator();
        while (iterator.hasNext() && !(mm = Time.match(p = (String)iterator.next(), input)).isPresent()) {
        }
        if (mm.isPresent()) {
            GregorianCalendar result;
            int month;
            Matcher m = (Matcher)mm.get();
            String tz = m.group("tz");
            int year = Integer.parseInt(m.group("year"));
            int day = Integer.parseInt(m.group("day"));
            String monthS = m.group("month");
            if (monthS.matches("\\d+")) {
                month = Integer.parseInt(monthS) - 1;
            } else {
                try {
                    month = new SimpleDateFormat("yyyy-MMM-dd", Locale.ROOT).parse("2015-" + monthS + "-15").getMonth();
                }
                catch (ParseException e) {
                    return Maybe.absent("Unknown date format '" + input + "': invalid month '" + monthS + "'; try http://yaml.org/type/timestamp.html format e.g. 2015-06-15 16:00:00 +0000");
                }
            }
            if (Strings.isNonBlank(tz)) {
                TimeZone tzz = null;
                String tzCode = m.group("tzCode");
                if (Strings.isNonBlank(tzCode)) {
                    tz = tzCode;
                }
                if (tz.matches("\\d+")) {
                    tz = "+" + tz;
                } else {
                    tzz = Time.getTimeZone(tz);
                }
                if (tzz == null) {
                    Maybe<Matcher> tmm = Time.match(" ?(?<tzH>(\\+|\\-||)\\d" + Time.optionally(DIGIT) + ")" + Time.optionally(Time.optionally(":") + Time.namedGroup("tzM", "\\d\\d")), tz);
                    if (tmm.isAbsent()) {
                        return Maybe.absent("Unknown date format '" + input + "': invalid timezone '" + tz + "'; try http://yaml.org/type/timestamp.html format e.g. 2015-06-15 16:00:00 +0000");
                    }
                    Matcher tm = tmm.get();
                    String tzM = tm.group("tzM");
                    int offset = (60 * Integer.parseInt(tm.group("tzH")) + Integer.parseInt("0" + (tzM != null ? tzM : ""))) * 60;
                    tzz = new SimpleTimeZone(offset * 1000, tz);
                }
                tz = Time.getTimeZoneOffsetString(tzz, year, month, day);
                result = new GregorianCalendar(tzz);
            } else {
                result = new GregorianCalendar();
            }
            result.clear();
            result.set(1, year);
            result.set(2, month);
            result.set(5, day);
            if (m.group("hours") != null) {
                String secsS;
                int hours = Integer.parseInt(m.group("hours"));
                String meridian = m.group("meridian");
                if (Strings.isNonBlank(meridian) && meridian.toLowerCase().startsWith("p")) {
                    if (hours > 12) {
                        return Maybe.absent("Unknown date format '" + input + "': can't be " + hours + " PM; try http://yaml.org/type/timestamp.html format e.g. 2015-06-15 16:00:00 +0000");
                    }
                    hours += 12;
                }
                result.set(11, hours);
                String minsS = m.group("mins");
                if (Strings.isNonBlank(minsS)) {
                    result.set(12, Integer.parseInt(minsS));
                }
                if (!Strings.isBlank(secsS = m.group("secs"))) {
                    if (secsS.matches("\\d\\d?")) {
                        result.set(13, Integer.parseInt(secsS));
                    } else {
                        double s = Double.parseDouble(secsS);
                        if (secsS.indexOf(46) < 0) {
                            if (secsS.length() == 5) {
                                s /= 1000.0;
                            } else {
                                return Maybe.absent("Unknown date format '" + input + "': invalid seconds '" + secsS + "'; try http://yaml.org/type/timestamp.html format e.g. 2015-06-15 16:00:00 +0000");
                            }
                        }
                        result.set(13, (int)s);
                        result.set(14, (int)Math.round(s * 1000.0) % 1000);
                    }
                }
            }
            return Maybe.of(result);
        }
        return Maybe.absent("Unknown date format '" + input + "'; try http://yaml.org/type/timestamp.html format e.g. 2015-06-15 16:00:00 +0000");
    }

    public static TimeZone getTimeZone(String code) {
        String[] timeZones;
        TimeZone tz;
        if (code.indexOf(47) == -1) {
            if ("Z".equals(code)) {
                return TIME_ZONE_UTC;
            }
            if ("UTC".equals(code)) {
                return TIME_ZONE_UTC;
            }
            if ("GMT".equals(code)) {
                return TIME_ZONE_UTC;
            }
            if ("EST".equals(code)) {
                return Time.getTimeZone("America/New_York");
            }
            if ("EDT".equals(code)) {
                return Time.getTimeZone("America/New_York");
            }
            if ("PST".equals(code)) {
                return Time.getTimeZone("America/Los_Angeles");
            }
            if ("PDT".equals(code)) {
                return Time.getTimeZone("America/Los_Angeles");
            }
            if ("CST".equals(code)) {
                return Time.getTimeZone("America/Chicago");
            }
            if ("CDT".equals(code)) {
                return Time.getTimeZone("America/Chicago");
            }
            if ("MST".equals(code)) {
                return Time.getTimeZone("America/Denver");
            }
            if ("MDT".equals(code)) {
                return Time.getTimeZone("America/Denver");
            }
            if ("BST".equals(code)) {
                return Time.getTimeZone("Europe/London");
            }
            if ("CEST".equals(code)) {
                return Time.getTimeZone("Europe/Paris");
            }
        }
        if ((tz = TimeZone.getTimeZone(code)) != null && !tz.equals(TimeZone.getTimeZone("GMT"))) {
            return tz;
        }
        for (String tzs : timeZones = TimeZone.getAvailableIDs()) {
            if (!tzs.equals(code)) continue;
            return tz;
        }
        return null;
    }

    public static Maybe<String> getTimeZoneOffsetString(String tz, int year, int month, int day) {
        TimeZone tzz = Time.getTimeZone(tz);
        if (tzz == null) {
            return Maybe.absent("Unknown time zone code: " + tz);
        }
        return Maybe.of(Time.getTimeZoneOffsetString(tzz, year, month, day));
    }

    public static String getTimeZoneOffsetString(TimeZone tz, int year, int month, int day) {
        int tzMins = tz.getOffset(new Date(year, month, day).getTime()) / 60 / 1000;
        String tzStr = (tzMins < 0 ? "-" : "+") + Strings.makePaddedString("" + Math.abs(tzMins) / 60, 2, "0", "") + Strings.makePaddedString("" + Math.abs(tzMins) % 60, 2, "0", "");
        return tzStr;
    }

    private static String namedGroup(String name, String pattern) {
        return "(?<" + name + ">" + pattern + ")";
    }

    private static String anyChar(String charSet) {
        return "[" + charSet + "]";
    }

    private static String optionally(String pattern) {
        return "(" + pattern + ")?";
    }

    private static String options(String ... patterns) {
        return "(" + Strings.join(patterns, "|") + ")";
    }

    private static String notMatching(String pattern) {
        return "(?!" + pattern + ")";
    }

    private static Maybe<Matcher> match(String pattern, String input) {
        Matcher m = Pattern.compile("^" + pattern + "$").matcher(input);
        if (m.find()) {
            return Maybe.of(m);
        }
        return Maybe.absent();
    }

    public static Maybe<Calendar> parseCalendarFormat(String dateString, String format) {
        return Time.parseCalendarFormat(dateString, new SimpleDateFormat(format, Locale.ROOT));
    }

    public static Maybe<Calendar> parseCalendarFormat(String dateString, DateFormat format) {
        if (dateString == null) {
            return Maybe.absent((Supplier<? extends RuntimeException>)((Supplier)() -> new NumberFormatException("GeneralHelper.parseDateString cannot parse a null string")));
        }
        try {
            Preconditions.checkNotNull((Object)format, (Object)"date format");
            dateString = dateString.trim();
            ParsePosition p = new ParsePosition(0);
            Date result = format.parse(dateString, p);
            if (result != null) {
                return Maybe.of(Time.newCalendarFromDate(result));
            }
            if (log.isTraceEnabled()) {
                log.trace("Could not parse date " + dateString + " using format " + format + ": " + p);
            }
            return Maybe.absent();
        }
        catch (Exception e2) {
            IllegalArgumentException e2;
            if (log.isTraceEnabled()) {
                e2 = new IllegalArgumentException("Could not parse date " + dateString + " using format " + format, e2);
            }
            return Maybe.absent(e2);
        }
    }

    public static Date dropMilliseconds(Date date) {
        return date == null ? null : (date.getTime() % 1000L != 0L ? new Date(date.getTime() - date.getTime() % 1000L) : date);
    }

    public static Duration elapsedSince(long timestamp) {
        return Duration.millis(System.currentTimeMillis() - timestamp);
    }

    public static boolean hasElapsedSince(long timestamp, Duration duration) {
        return Time.elapsedSince(timestamp).compareTo(duration) > 0;
    }

    public static long now() {
        return System.currentTimeMillis();
    }

    private static final class FunctionDurationToTimeStringRounded
    implements Function<Duration, String> {
        private FunctionDurationToTimeStringRounded() {
        }

        @Nullable
        public String apply(@Nullable Duration input) {
            if (input == null) {
                return null;
            }
            return Time.makeTimeStringRounded(input);
        }
    }

    private static final class FunctionLongToTimeStringRounded
    implements Function<Long, String> {
        private FunctionLongToTimeStringRounded() {
        }

        @Nullable
        public String apply(@Nullable Long input) {
            if (input == null) {
                return null;
            }
            return Time.makeTimeStringRounded(input);
        }
    }

    private static final class FunctionLongToTimeStringExact
    implements Function<Long, String> {
        private FunctionLongToTimeStringExact() {
        }

        @Nullable
        public String apply(@Nullable Long input) {
            if (input == null) {
                return null;
            }
            return Time.makeTimeStringExact(input);
        }
    }
}

