/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.shell.commands;

import com.google.common.annotations.VisibleForTesting;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Pattern;
import org.apache.accumulo.core.client.AccumuloClient;
import org.apache.accumulo.core.client.NamespaceNotFoundException;
import org.apache.accumulo.core.client.admin.TableOperations;
import org.apache.accumulo.core.clientImpl.ClientContext;
import org.apache.accumulo.core.clientImpl.Namespaces;
import org.apache.accumulo.core.data.NamespaceId;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.core.metadata.schema.DataFileValue;
import org.apache.accumulo.core.metadata.schema.TabletMetadata;
import org.apache.accumulo.core.metadata.schema.TabletsMetadata;
import org.apache.accumulo.core.util.NumUtil;
import org.apache.accumulo.shell.Shell;
import org.apache.accumulo.shell.commands.OptUtil;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.hadoop.io.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ListTabletsCommand
extends Shell.Command {
    private static final Logger log = LoggerFactory.getLogger(ListTabletsCommand.class);
    private Option outputFileOpt;
    private Option optTablePattern;
    private Option optHumanReadable;
    private Option optNamespace;
    private Option disablePaginationOpt;

    @Override
    public int execute(String fullCommand, CommandLine cl, Shell shellState) throws Exception {
        Set<TableInfo> tableInfoSet = this.populateTables(cl, shellState);
        if (tableInfoSet.isEmpty()) {
            log.warn("No tables found that match your criteria");
            return 0;
        }
        boolean humanReadable = cl.hasOption(this.optHumanReadable.getOpt());
        LinkedList<String> lines = new LinkedList<String>();
        lines.add(TabletRowInfo.header);
        for (TableInfo tableInfo : tableInfoSet) {
            String name = tableInfo.name;
            lines.add("TABLE: " + name);
            List<TabletRowInfo> rows = this.getTabletRowInfo(shellState, tableInfo);
            for (int i = 0; i < rows.size(); ++i) {
                TabletRowInfo row = rows.get(i);
                lines.add(row.format(i + 1, humanReadable));
            }
        }
        if (lines.size() == 1) {
            lines.add("No data");
        }
        this.printResults(cl, shellState, lines);
        return 0;
    }

    @VisibleForTesting
    protected void printResults(CommandLine cl, Shell shellState, List<String> lines) throws Exception {
        if (cl.hasOption(this.outputFileOpt.getOpt())) {
            String outputFile = cl.getOptionValue(this.outputFileOpt.getOpt());
            Shell.PrintFile printFile = new Shell.PrintFile(outputFile);
            shellState.printLines(lines.iterator(), false, printFile);
            printFile.close();
        } else {
            boolean paginate = !cl.hasOption(this.disablePaginationOpt.getOpt());
            shellState.printLines(lines.iterator(), paginate);
        }
    }

    private Set<TableInfo> populateTables(CommandLine cl, Shell shellState) throws NamespaceNotFoundException {
        TableOperations tableOps = shellState.getAccumuloClient().tableOperations();
        Map tableIdMap = tableOps.tableIdMap();
        TreeSet<TableInfo> tableSet = new TreeSet<TableInfo>();
        if (cl.hasOption(this.optTablePattern.getOpt())) {
            Pattern tablePattern = Pattern.compile(cl.getOptionValue(this.optTablePattern.getOpt()));
            for (String table : tableOps.list()) {
                if (!tablePattern.matcher(table).matches()) continue;
                TableId id = TableId.of((String)((String)tableIdMap.get(table)));
                tableSet.add(new TableInfo(table, id));
            }
            return tableSet;
        }
        if (cl.hasOption(this.optNamespace.getOpt())) {
            String nsName = cl.getOptionValue(this.optNamespace.getOpt());
            NamespaceId namespaceId = Namespaces.getNamespaceId((ClientContext)shellState.getContext(), (String)nsName);
            List tables = Namespaces.getTableNames((ClientContext)shellState.getContext(), (NamespaceId)namespaceId);
            tables.forEach(name -> {
                String tableIdString = (String)tableIdMap.get(name);
                if (tableIdString != null) {
                    TableId id = TableId.of((String)tableIdString);
                    tableSet.add(new TableInfo((String)name, id));
                } else {
                    log.warn("Table not found: {}", name);
                }
            });
            return tableSet;
        }
        if (cl.hasOption("t")) {
            String table = cl.getOptionValue("t");
            String idString = (String)tableIdMap.get(table);
            if (idString != null) {
                TableId id = TableId.of((String)idString);
                tableSet.add(new TableInfo(table, id));
            } else {
                log.warn("Table not found: {}", (Object)table);
            }
            return tableSet;
        }
        String table = shellState.getTableName();
        if (!table.isEmpty()) {
            TableId id = TableId.of((String)((String)tableIdMap.get(table)));
            tableSet.add(new TableInfo(table, id));
            return tableSet;
        }
        return Collections.emptySet();
    }

    private List<TabletRowInfo> getTabletRowInfo(Shell shellState, TableInfo tableInfo) throws Exception {
        log.trace("scan metadata for tablet info table name: '{}', tableId: '{}' ", (Object)tableInfo.name, (Object)tableInfo.id);
        List<TabletRowInfo> tResults = this.getMetadataInfo(shellState, tableInfo);
        if (log.isTraceEnabled()) {
            for (TabletRowInfo tabletRowInfo : tResults) {
                log.trace("Tablet info: {}", (Object)tabletRowInfo);
            }
        }
        return tResults;
    }

    protected List<TabletRowInfo> getMetadataInfo(Shell shellState, TableInfo tableInfo) throws Exception {
        ArrayList<TabletRowInfo> results = new ArrayList<TabletRowInfo>();
        ClientContext context = shellState.getContext();
        Set liveTserverSet = TabletMetadata.getLiveTServers((ClientContext)context);
        try (TabletsMetadata tabletsMetadata = TabletsMetadata.builder((AccumuloClient)context).forTable(tableInfo.id).build();){
            for (TabletMetadata md : tabletsMetadata) {
                TabletRowInfo.Factory factory = new TabletRowInfo.Factory(tableInfo.name, md.getExtent());
                Map fileMap = md.getFilesMap();
                factory.numFiles(fileMap.size());
                long entries = 0L;
                long size = 0L;
                for (DataFileValue dfv : fileMap.values()) {
                    entries += dfv.getNumEntries();
                    size += dfv.getSize();
                }
                factory.numEntries(entries);
                factory.size(size);
                factory.numWalLogs(md.getLogs().size());
                factory.dir(md.getDirName());
                factory.location(md.getLocation());
                factory.status(md.getTabletState(liveTserverSet).toString());
                results.add(factory.build());
            }
        }
        return results;
    }

    @Override
    public String description() {
        return "Prints info about every tablet for a table, one tablet per line.";
    }

    @Override
    public int numArgs() {
        return 0;
    }

    @Override
    public Options getOptions() {
        Options opts = new Options();
        opts.addOption(OptUtil.tableOpt("table to be scanned"));
        this.optTablePattern = new Option("p", "pattern", true, "regex pattern of table names");
        this.optTablePattern.setArgName("pattern");
        opts.addOption(this.optTablePattern);
        this.optNamespace = new Option("ns", "namespace", true, "name of a namespace");
        this.optNamespace.setArgName("namespace");
        opts.addOption(this.optNamespace);
        this.optHumanReadable = new Option("h", "human-readable", false, "format large sizes to human readable units");
        this.optHumanReadable.setArgName("human readable output");
        opts.addOption(this.optHumanReadable);
        this.disablePaginationOpt = new Option("np", "no-pagination", false, "disables pagination of output");
        opts.addOption(this.disablePaginationOpt);
        this.outputFileOpt = new Option("o", "output", true, "local file to write output to");
        this.outputFileOpt.setArgName("file");
        opts.addOption(this.outputFileOpt);
        return opts;
    }

    static class TabletRowInfo {
        public final String tableName;
        public final int numFiles;
        public final int numWalLogs;
        public final long numEntries;
        public final long size;
        public final String status;
        public final String location;
        public final String dir;
        public final TableId tableId;
        public final KeyExtent tablet;
        public final boolean tableExists;
        public static final String header = String.format("%-4s %-15s %-5s %-5s %-9s %-9s %-10s %-30s %-5s %-20s %-20s", "NUM", "TABLET_DIR", "FILES", "WALS", "ENTRIES", "SIZE", "STATUS", "LOCATION", "ID", "START (Exclusive)", "END");

        private TabletRowInfo(String tableName, KeyExtent tablet, int numFiles, int numWalLogs, long numEntries, long size, String status, String location, String dir, boolean tableExists) {
            this.tableName = tableName;
            this.tableId = tablet.tableId();
            this.tablet = tablet;
            this.numFiles = numFiles;
            this.numWalLogs = numWalLogs;
            this.numEntries = numEntries;
            this.size = size;
            this.status = status;
            this.location = location;
            this.dir = dir;
            this.tableExists = tableExists;
        }

        String getNumEntries(boolean humanReadable) {
            if (humanReadable) {
                return String.format("%9s", NumUtil.bigNumberForQuantity((long)this.numEntries));
            }
            return Long.toString(this.numEntries);
        }

        String getSize(boolean humanReadable) {
            if (humanReadable) {
                return String.format("%9s", NumUtil.bigNumberForSize((long)this.size));
            }
            return Long.toString(this.size);
        }

        public String getEndRow() {
            Text t = this.tablet.endRow();
            if (t == null) {
                return "+INF";
            }
            return t.toString();
        }

        public String getStartRow() {
            Text t = this.tablet.prevEndRow();
            if (t == null) {
                return "-INF";
            }
            return t.toString();
        }

        String format(int number, boolean prettyPrint) {
            return String.format("%-4d %-15s %-5d %-5s %-9s %-9s %-10s %-30s %-5s %-20s %-20s", number, this.dir, this.numFiles, this.numWalLogs, this.getNumEntries(prettyPrint), this.getSize(prettyPrint), this.status, this.location, this.tableId, this.getStartRow(), this.getEndRow());
        }

        public String getTablet() {
            return this.getStartRow() + " " + this.getEndRow();
        }

        public static class Factory {
            final String tableName;
            final KeyExtent tablet;
            final TableId tableId;
            int numFiles = 0;
            int numWalLogs = 0;
            long numEntries = 0L;
            long size = 0L;
            String status = "";
            String location = "";
            String dir = "";
            boolean tableExists = false;

            Factory(String tableName, KeyExtent tablet) {
                this.tableName = tableName;
                this.tablet = tablet;
                this.tableId = tablet.tableId();
            }

            Factory numFiles(int numFiles) {
                this.numFiles = numFiles;
                return this;
            }

            Factory numWalLogs(int numWalLogs) {
                this.numWalLogs = numWalLogs;
                return this;
            }

            public Factory numEntries(long numEntries) {
                this.numEntries = numEntries;
                return this;
            }

            public Factory size(long size) {
                this.size = size;
                return this;
            }

            public Factory status(String status) {
                this.status = status;
                return this;
            }

            public Factory location(TabletMetadata.Location location) {
                if (location == null) {
                    this.location = "None";
                } else {
                    String server = location.getHostPort();
                    this.location = location.getType() + ":" + server;
                }
                return this;
            }

            public Factory dir(String dirName) {
                this.dir = dirName;
                return this;
            }

            public Factory tableExists(boolean tableExists) {
                this.tableExists = tableExists;
                return this;
            }

            public TabletRowInfo build() {
                return new TabletRowInfo(this.tableName, this.tablet, this.numFiles, this.numWalLogs, this.numEntries, this.size, this.status, this.location, this.dir, this.tableExists);
            }
        }
    }

    static class TableInfo
    implements Comparable<TableInfo> {
        public final String name;
        public final TableId id;

        public TableInfo(String name, TableId id) {
            this.name = name;
            this.id = id;
        }

        @Override
        public int compareTo(TableInfo other) {
            return this.name.compareTo(other.name);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            TableInfo tableInfo = (TableInfo)o;
            return this.name.equals(tableInfo.name);
        }

        public int hashCode() {
            return Objects.hash(this.name);
        }
    }
}

